package io.ghostwriter.rt.snaperr.serializer;


import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import io.ghostwriter.rt.snaperr.trigger.Trigger;
import io.ghostwriter.rt.snaperr.trigger.WatchedValue;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

public class JsonSerializer implements TriggerSerializer<String> {

    // We use a Representation (API) class to avoid leaking implementation details
    private class Representation {

        private final Object context;

        private final String methodName;

        private final Map<String, Object> stateSnapshot;

        private final String cause;

        private final StackTraceElement[] stackTrace;

        public Representation(Object context, String methodName, Throwable throwable, Map<String, Object> state) {
            this.context = context;
            this.methodName = methodName;
            this.stateSnapshot = state;
            this.cause = String.valueOf(throwable);
            this.stackTrace = limitedStackTrace(throwable);
        }

        private StackTraceElement[] limitedStackTrace(Throwable throwable) {
            boolean stackTraceLimitDisabled = stackTraceLimit == null || stackTraceLimit == 0;
            final StackTraceElement[] stackTrace = throwable.getStackTrace();
            if (stackTraceLimitDisabled) {
                return stackTrace;
            }

            return Arrays.copyOfRange(stackTrace, 0, stackTraceLimit);
        }
    }

    private final Gson gson;

    private final Integer stackTraceLimit;

    public JsonSerializer() {
        this(true, null);
    }

    public JsonSerializer(boolean doPrettyPrint, Integer stackTraceLimit) {
        GsonBuilder gsonBuilder = new GsonBuilder();
        gsonBuilder.enableComplexMapKeySerialization();
        if (doPrettyPrint) {
            gsonBuilder.setPrettyPrinting();
        }

        gson = gsonBuilder.create();
        if (stackTraceLimit < 0) {
            throw new IllegalArgumentException("Stack trace limit must be a positive number! Got: " + stackTraceLimit);
        }
        this.stackTraceLimit = stackTraceLimit;
    }

    @Override
    public String serializeTrigger(Trigger trigger) {
        Objects.requireNonNull(trigger);


        Object context = trigger.getContext();
        // in case of static methods, GhostWriter passes the Class<?> reference
        if (context instanceof Class<?>) {
            // GSON cannot serialize java.lang.Class so we turn it into a String
            Class<?> klazz = (Class<?>) context;
            context = klazz.getCanonicalName();
        }

        final String methodName = trigger.getMethodName();
        final Throwable throwable = trigger.getThrowable();
        final Map<String, Object> state = getState(trigger.getWatched());

        Representation representation = new Representation(context, methodName, throwable, state);

        final String json = gson.toJson(representation);

        return json;
    }

    private Map<String, Object> getState(Map<String, WatchedValue> watched) {
        Map<String, Object> state = new HashMap<>(watched.size());

        for (Map.Entry<String, WatchedValue> element : watched.entrySet())  {
            final String variableName = element.getKey();
            final Object elementValue = element.getValue().getValue();

            state.put(variableName, elementValue);
        }

        return state;
    }

}
