package io.ghostwriter.rt.snaperr;

import io.ghostwriter.Tracer;
import io.ghostwriter.rt.snaperr.handler.Slf4jWriter;
import io.ghostwriter.rt.snaperr.trigger.Trigger;
import io.ghostwriter.rt.snaperr.trigger.TriggerHandler;
import io.ghostwriter.rt.snaperr.trigger.WatchedValue;

import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;

public class GhostWriterSnaperr implements Tracer {

    private ThreadLocal<Map<String, WatchedValue>> watchedThreadState = new ThreadLocal<Map<String, WatchedValue>>() {
        @Override
        protected Map<String, WatchedValue> initialValue() {
            return new TreeMap<>();
        }
    };

    // We store the triggered error, since we use it to avoid triggering additional error events on
    // the same thread. This contradicts the improvements described in issue #57.
    private ThreadLocal<Trigger> processedTrigger = new ThreadLocal<Trigger>() {
        @Override
        protected Trigger initialValue() {
            return null;
        }
    };

    private TriggerHandler triggerHandler;

    static private TriggerHandler defaultTriggerHandler() {
        return new Slf4jWriter();
    }

    public GhostWriterSnaperr() {
        this(defaultTriggerHandler());
    }

    public GhostWriterSnaperr(TriggerHandler triggerHandler) {
        this.triggerHandler = Objects.requireNonNull(triggerHandler);
    }

    @Override
    public void entering(Object source, String method, Object... params) {
        // Currently we only track the current method's scope and state changes.
        // It would be possible to keep the stack, for details check the issue #57
        final Map<String, WatchedValue> watched = watchedThreadState.get();
        watched.clear();
        // if we have an entered event, it means that everything is fine
        clearErrorState();

        for (int i = 0; i < params.length - 1; i++) {
            final Object paramName = params[i++];
            final Object paramValue = params[i];
            final String name = (String) paramName;
            watch(name, paramValue);
        }
    }

    @Override
    public void exiting(Object source, String method, Object returnValue) {
        final Map<String, WatchedValue> watched = watchedThreadState.get();
        watched.clear(); // we successfully exited, thus can drop the current scope's references
    }

    @Override
    public void exiting(Object source, String method) {
        final Map<String, WatchedValue> watched = watchedThreadState.get();
        watched.clear();
    }

    @Override
    public void valueChange(Object source, String method, String variable, Object value) {
        watch(variable, value);
    }

    @Override
    public void onError(Object source, String method, Throwable error) {
        if (isPropagatedError()) {
            return;
        }

        // The trigger handler executes in the same thread as the one that just "crashed".
        // This way we can guarantee that the watched references are not changed. At least not by the current thread.
        // It is possible that the snapshot contains a reference to a global state that could be modified by other threads.
        // However that can _only_ (I'm like 80% sure!) happen when the original code already has race conditions.
        Trigger trigger = new Trigger(source, method, watchedThreadState.get(), error);
        setErrorState(trigger);
        if (triggerHandler != null) {
            triggerHandler.onError(trigger);
        }
    }

    private void watch(String variableName, Object variableReference) {
        final Map<String, WatchedValue> watched = watchedThreadState.get();
        WatchedValue watchedValue = new WatchedValue(variableName, variableReference);
        watched.put(variableName, watchedValue);
    }

    /**
     * Set Snapper in a state where it is ready to trigger in case an unexpected error occurs.
     */
    private void clearErrorState() {
        processedTrigger.set(null);
    }

    private void setErrorState(Trigger trigger) {
        processedTrigger.set(trigger);
    }

    private boolean isPropagatedError() {
        return processedTrigger.get() != null;
    }

}
