package io.geewit.data.jpa.envers.repository;

import java.io.Serializable;

import javax.persistence.EntityManager;

import io.geewit.data.jpa.envers.repository.impl.EnversRevisionRepositoryImpl;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.support.JpaEntityInformation;
import org.springframework.data.jpa.repository.support.JpaRepositoryFactory;
import org.springframework.data.jpa.repository.support.JpaRepositoryFactoryBean;
import org.springframework.data.repository.core.RepositoryInformation;
import org.springframework.data.repository.core.RepositoryMetadata;
import org.springframework.data.repository.core.support.RepositoryComposition;
import org.springframework.data.repository.core.support.RepositoryFactorySupport;


/**
 * {@link FactoryBean} creating {@link EnversRevisionRepository} instances.
 *
 * @author Oliver Gierke
 * @author Michael Igler
 */
@SuppressWarnings({"unused"})
public class EnversRevisionRepositoryFactoryBean<T extends EnversRevisionRepository<S, ID, O>, S, ID, O extends Serializable> extends JpaRepositoryFactoryBean<T, S, ID> {

	/**
	 * Creates a new {@link EnversRevisionRepositoryFactoryBean} for the given repository interface.
	 *
	 * @param repositoryInterface must not be {@literal null}.
	 */
	public EnversRevisionRepositoryFactoryBean(Class<? extends T> repositoryInterface) {
		super(repositoryInterface);
	}


	/**
	 * @see org.springframework.data.jpa.repository.support.JpaRepositoryFactoryBean#createRepositoryFactory(javax.persistence.EntityManager)
	 */
	@Override
	protected RepositoryFactorySupport createRepositoryFactory(EntityManager entityManager) {
		return new RevisionRepositoryFactory<>(entityManager);
	}

	/**
	 * Repository factory creating {@link EnversRevisionRepository} instances.
	 *
	 * @author Oliver Gierke
	 */
	private static class RevisionRepositoryFactory<T, ID, O extends Serializable> extends JpaRepositoryFactory {

		private final EnversRevisionEntityInformation revisionEntityInformation;
		private final EntityManager entityManager;

		/**
		 * Creates a new {@link RevisionRepositoryFactory} using the given {@link EntityManager} and revision entity class.
		 *
		 * @param entityManager must not be {@literal null}.
		 */
		public RevisionRepositoryFactory(EntityManager entityManager) {

			super(entityManager);
			this.entityManager = entityManager;
			this.revisionEntityInformation = new EnversRevisionEntityInformation();
		}

		/**
		 * Callback to create a {@link JpaRepository} instance with the given {@link EntityManager}
		 *
		 * @param information will never be {@literal null}.
		 * @param entityManager will never be {@literal null}.
		 * @return
		 */
		@Override
		protected EnversRevisionRepositoryImpl<T, ID, O> getTargetRepository(RepositoryInformation information,
																		EntityManager entityManager) {

			JpaEntityInformation<T, ID> entityInformation = (JpaEntityInformation<T, ID>) getEntityInformation(information.getDomainType());

			return new EnversRevisionRepositoryImpl<>(entityInformation, revisionEntityInformation, entityManager);
		}

		/**
		 * @see org.springframework.data.jpa.repository.support.JpaRepositoryFactory#getRepositoryBaseClass(org.springframework.data.repository.core.RepositoryMetadata)
		 */
		@Override
		protected Class<?> getRepositoryBaseClass(RepositoryMetadata metadata) {
			return EnversRevisionRepositoryImpl.class;
		}

		/**
		 * @see org.springframework.data.repository.core.support.RepositoryFactorySupport#getRepository(java.lang.Class, java.lang.Object)
		 */
		@SuppressWarnings("hiding")
		@Override
		public <I> I getRepository(Class<I> repositoryInterface, RepositoryComposition.RepositoryFragments fragments) {

			if (EnversRevisionRepository.class.isAssignableFrom(repositoryInterface)) {
				if (!revisionEntityInformation.getRevisionNumberType().equals(Integer.class)) {
					throw new IllegalStateException(String.format(
							"Configured a revision entity type of %s with a revision type of %s "
									+ "but the repository interface is typed to a revision type of %s!",
							repositoryInterface, revisionEntityInformation.getRevisionNumberType(), Integer.class));
				}
			}

			return super.getRepository(repositoryInterface, fragments);
		}
	}
}
