package io.gamedock.sdk.ads.utils.device;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.res.Configuration;
import android.os.Build;
import android.os.RemoteException;
import android.util.DisplayMetrics;

import com.android.installreferrer.api.InstallReferrerClient;
import com.android.installreferrer.api.InstallReferrerStateListener;
import com.android.installreferrer.api.ReferrerDetails;
import com.google.android.gms.ads.identifier.AdvertisingIdClient;

import java.util.concurrent.Callable;

import io.gamedock.sdk.ads.GamedockAds;
import io.gamedock.sdk.ads.utils.logging.LoggingUtilAds;
import io.reactivex.Observable;

/**
 * Class used to retrieve various device specific information.
 */
public class DeviceInfo {
    private static String advertisingId;
    private static String packageName;

    private static int deviceWidth;
    private static int deviceHeight;
    private static String deviceDensity;
    private static float deviceDensityValue;

    private static String referrerUrl;

    private static String os;
    private static String osVersion;
    private static String deviceManufacturer;
    private static String deviceModel;
    private static int deviceType;

    private static String vastVersion;

    private static String vastMinDuration;
    private static String vastMaxDuration;
    private static String vastFormatType;

    /**
     * Method used to retrieve the advertising id if possible.
     * @param context
     * @return
     */
    public static Observable<String> retrieveAdvertisingId(final Context context) {
        return Observable.fromCallable(new Callable<String>() {
            @Override
            public String call() throws Exception {
                try {
                    String advertisingId = AdvertisingIdClient.getAdvertisingIdInfo(context).getId();
                    LoggingUtilAds.d("Retrieved Advertising Id: " + advertisingId);
                    DeviceInfo.setAdvertisingId(advertisingId);
                } catch (Exception e) {
                    e.printStackTrace();
                }

                return "";
            }
        });
    }

    /**
     * Method used to retrieve the device specific sizes (width, height and density).
     * @param context
     * @return
     */
    public static Observable<String> retrieveDeviceSizes(final Context context) {
        return Observable.fromCallable(new Callable<String>() {
            @Override
            public String call() throws Exception {
                try {
                    DisplayMetrics displayMetrics = context.getResources().getDisplayMetrics();
                    setDeviceWidth(displayMetrics.widthPixels);
                    setDeviceHeight(displayMetrics.heightPixels);

                    LoggingUtilAds.d("Retrieved Device Width: " + displayMetrics.widthPixels);
                    LoggingUtilAds.d("Retrieved Device Height: " + displayMetrics.heightPixels);

                    int densityDpi = displayMetrics.densityDpi;
                    String density = null;
                    float densityValue = displayMetrics.density;

                    switch (densityDpi) {
                        case DisplayMetrics.DENSITY_LOW:
                            density = "LDPI";
                            break;
                        case DisplayMetrics.DENSITY_MEDIUM:
                            density = "MDPI";
                            break;
                        case DisplayMetrics.DENSITY_TV:
                        case DisplayMetrics.DENSITY_HIGH:
                            density = "HDPI";
                            break;
                        case DisplayMetrics.DENSITY_XHIGH:
                        case DisplayMetrics.DENSITY_280:
                            density = "XHDPI";
                            break;
                        case DisplayMetrics.DENSITY_XXHIGH:
                        case DisplayMetrics.DENSITY_360:
                        case DisplayMetrics.DENSITY_400:
                        case DisplayMetrics.DENSITY_420:
                            density = "XXHDPI";
                            break;
                        case DisplayMetrics.DENSITY_XXXHIGH:
                        case DisplayMetrics.DENSITY_560:
                            density = "XXXHDPI";
                            break;
                    }

                    setDeviceDensity(density);
                    setDeviceDensityValue(densityValue);

                    LoggingUtilAds.d("Retrieved Device Density: " + density);
                    LoggingUtilAds.d("Retrieved Device Density Value: " + densityValue);
                } catch (Exception e) {
                    e.printStackTrace();
                }

                return "";
            }
        });
    }

    /**
     * Method used to retrieve the package name for the app implementing the ads module.
     * @param context
     */
    public static void retrievePackageName(Context context) {
        try {
            setOs("android");
            setOsVersion(Build.VERSION.RELEASE);
            setPackageName(context.getPackageName());
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Method used to retrieve device specs such as model and manufacturer.
     */
    public static void retrieveDeviceSpecs(Context context) {
        try {
            setDeviceManufacturer(Build.MANUFACTURER);
            setDeviceModel(Build.MODEL);
            setVastVersion("4.0");
            setVastMinDuration("10");
            setVastMaxDuration("45");
            setVastFormatType("mp4");

            if (isTablet(context)) {
                setDeviceType(5);
            } else {
                setDeviceType(4);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Method used to retrieve the referer url with which the app was installed if present.
     * @param context
     * @return
     */
    public static Observable<String> retrieveRefererUrl(final Context context) {
        return Observable.fromCallable(new Callable<String>() {
            @Override
            public String call() throws Exception {
                try {
                    final SharedPreferences sharedPref = context.getSharedPreferences("GamedockAds", Context.MODE_PRIVATE);
                    String referrerUrlSaved = sharedPref.getString("GamedockAds-Referrer", null);
                    if (referrerUrlSaved != null) {
                        setReferrerUrl(referrerUrlSaved);
                        return "";
                    }

                    final InstallReferrerClient referrerClient = InstallReferrerClient.newBuilder(context).build();
                    referrerClient.startConnection(new InstallReferrerStateListener() {
                        @Override
                        public void onInstallReferrerSetupFinished(int responseCode) {
                            switch (responseCode) {
                                case InstallReferrerClient.InstallReferrerResponse.OK:
                                    try {
                                        ReferrerDetails response = referrerClient.getInstallReferrer();
                                        if (response != null) {
                                            String referrerUrl = response.getInstallReferrer();

                                            if (referrerUrl != null) {
                                                setReferrerUrl(referrerUrl);
                                                sharedPref.edit().putString("GamedockAds-Referrer", referrerUrl).apply();

                                                LoggingUtilAds.d("Retrieved Install Referrer: " + referrerUrl);
                                            }
                                        }

                                        referrerClient.endConnection();
                                    } catch (RemoteException e) {
                                        e.printStackTrace();
                                    }
                                    break;
                                case InstallReferrerClient.InstallReferrerResponse.FEATURE_NOT_SUPPORTED:
                                    // API not available on the current Play Store app.
                                    break;
                                case InstallReferrerClient.InstallReferrerResponse.SERVICE_UNAVAILABLE:
                                    // Connection couldn't be established.
                                    break;
                            }
                        }

                        @Override
                        public void onInstallReferrerServiceDisconnected() {
                        }
                    });
                } catch (Exception e) {
                    e.printStackTrace();
                }
                return "";
            }
        });
    }

    public static boolean isTablet(Context context) {
        return (context.getResources().getConfiguration().screenLayout & Configuration.SCREENLAYOUT_SIZE_MASK) >= Configuration.SCREENLAYOUT_SIZE_LARGE;
    }

    public static String getAdvertisingId() {
        if (!GamedockAds.getInstance().getSettings().isCOPPACompliant()) {
            return advertisingId;
        } else {
            return "";
        }
    }

    public static void setAdvertisingId(String advertisingId) {
        DeviceInfo.advertisingId = advertisingId;
    }

    public static String getPackageName() {
        return packageName;
    }

    public static void setPackageName(String packageName) {
        DeviceInfo.packageName = packageName;
    }

    public static int getDeviceWidth() {
        return deviceWidth;
    }

    public static void setDeviceWidth(int deviceWidth) {
        DeviceInfo.deviceWidth = deviceWidth;
    }

    public static int getDeviceHeight() {
        return deviceHeight;
    }

    public static void setDeviceHeight(int deviceHeight) {
        DeviceInfo.deviceHeight = deviceHeight;
    }

    public static String getDeviceDensity() {
        return deviceDensity;
    }

    public static void setDeviceDensity(String deviceDensity) {
        DeviceInfo.deviceDensity = deviceDensity;
    }

    public static float getDeviceDensityValue() {
        return deviceDensityValue;
    }

    public static void setDeviceDensityValue(float deviceDensityValue) {
        DeviceInfo.deviceDensityValue = deviceDensityValue;
    }

    public static String getReferrerUrl() {
        return referrerUrl;
    }

    public static void setReferrerUrl(String referrerUrl) {
        DeviceInfo.referrerUrl = referrerUrl;
    }

    public static String getOs() {
        return os;
    }

    public static void setOs(String os) {
        DeviceInfo.os = os;
    }

    public static String getOsVersion() {
        return osVersion;
    }

    public static void setOsVersion(String osVersion) {
        DeviceInfo.osVersion = osVersion;
    }

    public static String getDeviceManufacturer() {
        return deviceManufacturer;
    }

    public static void setDeviceManufacturer(String deviceManufacturer) {
        DeviceInfo.deviceManufacturer = deviceManufacturer;
    }

    public static String getDeviceModel() {
        return deviceModel;
    }

    public static void setDeviceModel(String deviceModel) {
        DeviceInfo.deviceModel = deviceModel;
    }

    public static int getDeviceType() {
        return deviceType;
    }

    public static void setDeviceType(int deviceType) {
        DeviceInfo.deviceType = deviceType;
    }

    public static String getVastVersion() {
        return vastVersion;
    }

    public static void setVastVersion(String vastVersion) {
        DeviceInfo.vastVersion = vastVersion;
    }

    public static String getVastMinDuration() {
        return vastMinDuration;
    }

    public static void setVastMinDuration(String vastMinDuration) {
        DeviceInfo.vastMinDuration = vastMinDuration;
    }

    public static String getVastMaxDuration() {
        return vastMaxDuration;
    }

    public static void setVastMaxDuration(String vastMaxDuration) {
        DeviceInfo.vastMaxDuration = vastMaxDuration;
    }

    public static String getVastFormatType() {
        return vastFormatType;
    }

    public static void setVastFormatType(String vastFormatType) {
        DeviceInfo.vastFormatType = vastFormatType;
    }
}
