package io.gamedock.sdk.ads.core.request;

import android.content.Context;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Scanner;

import javax.net.ssl.HttpsURLConnection;

import io.gamedock.sdk.ads.utils.error.ErrorCodes;
import io.gamedock.sdk.ads.utils.logging.LoggingUtilAds;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.Observer;
import io.reactivex.annotations.NonNull;
import io.reactivex.disposables.Disposable;

/**
 * Base class containing general logic for all the requests.
 */
public abstract class BaseRequest {
    protected Context context;

    protected AdType adType;
    protected RequestType requestType;
    protected RequestListener requestListener;

    public BaseRequest(Context context, AdType adType, RequestType requestType, RequestListener requestListener) {
        this.context = context;
        this.adType = adType;
        this.requestType = requestType;
        this.requestListener = requestListener;
    }

    /**
     * Observer object used to relay back information from the request using the attached callback listener.
     */
    public Observer<String> requestObserver = new Observer<String>() {
        @Override
        public void onSubscribe(@NonNull Disposable d) {

        }

        @Override
        public void onNext(@NonNull String responseJSON) {
            if (requestListener != null) {
                requestListener.onSuccess(responseJSON);
            }
        }

        @Override
        public void onError(@NonNull Throwable e) {
            e.printStackTrace();
            if (requestListener != null) {
                requestListener.onFailure(ErrorCodes.ConnectionError);
            }

        }

        @Override
        public void onComplete() {

        }
    };

    /**
     * Method used to make a network request observable.
     * @return
     */
    public Observable<String> loadRequest() {
        final BaseRequest request = this;
        return Observable.create(new ObservableOnSubscribe<String>() {
            @Override
            public void subscribe(ObservableEmitter<String> observableEmitter) throws Exception {
                try {
                    String response = request.makeRequestCall();
                    observableEmitter.onNext(response);
                    observableEmitter.onComplete();
                } catch (Exception e) {
                    observableEmitter.onError(e);
                }
            }
        });
    }

    /**
     * Method used to make the network request to the ad provider backend.
     * @return
     * @throws Exception
     */
    private String makeRequestCall() throws Exception {
        String response = "";
        HttpsURLConnection connection = null;
        String errorResponse = null;
        int responseCode = 0;

        try {
            String urlValue = getUrl();

            if (urlValue == null || urlValue.isEmpty()) {
                LoggingUtilAds.d("Request url is null or empty. Canceling request");
                throw new NullPointerException();
            }

            LoggingUtilAds.d("Making network request with Url: " + urlValue);

            URL url = new URL(urlValue);
            connection = (HttpsURLConnection) url.openConnection();
            connection.setRequestMethod("GET");
            connection.setReadTimeout(8000);
            connection.setConnectTimeout(4000);
            connection.setDoInput(true);

            responseCode = connection.getResponseCode();

            if (responseCode == HttpsURLConnection.HTTP_OK) {
                response = readInputStreamToString(connection);
            } else {
                InputStream stream = connection.getErrorStream();
                Scanner scanner = new Scanner(stream);
                scanner.useDelimiter("\\Z");
                errorResponse = scanner.next();

                if (responseCode == HttpsURLConnection.HTTP_UNAUTHORIZED) {
                    LoggingUtilAds.w("Response error: " + errorResponse);
                    return response;
                } else if (responseCode == HttpURLConnection.HTTP_BAD_GATEWAY
                        || responseCode == HttpURLConnection.HTTP_UNAVAILABLE
                        || responseCode == HttpURLConnection.HTTP_GATEWAY_TIMEOUT) {
                    throw new ConnectException("\n" + errorResponse);
                }
            }
        } catch (Exception e) {
            throw e;
        } finally {
            try {
                assert connection != null;
                connection.disconnect();
            } catch (Exception e) {
                e.printStackTrace(); //If you want further info on failure...
            }
        }
        return response;
    }

    /**
     * Method used to read and process the response from the connection input stream.
     * @param connection
     * @return
     */
    private String readInputStreamToString(HttpURLConnection connection) {
        String response = null;
        StringBuilder sb = new StringBuilder();
        InputStream is = null;

        try {
            is = new BufferedInputStream(connection.getInputStream());
            BufferedReader br = new BufferedReader(new InputStreamReader(is));
            String inputLine;
            while ((inputLine = br.readLine()) != null) {
                sb.append(inputLine);
            }
            response = sb.toString();
        } catch (Exception e) {
            LoggingUtilAds.i("Error reading InputStream");
            response = null;
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    LoggingUtilAds.i("Error closing InputStream");
                }
            }
        }
        return response;
    }

    public abstract String getUrl();
}
