package io.gamedock.sdk.ads.core.interstitial;

import android.content.Context;
import android.content.Intent;

import com.google.gson.Gson;

import io.gamedock.sdk.ads.GamedockAds;
import io.gamedock.sdk.ads.core.base.AdData;
import io.gamedock.sdk.ads.core.base.BaseAd;
import io.gamedock.sdk.ads.core.request.AdType;
import io.gamedock.sdk.ads.core.request.RequestListener;
import io.gamedock.sdk.ads.core.request.RequestType;
import io.gamedock.sdk.ads.core.ui.GamedockAdsActivity;
import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.ads.providers.improvedigital.ImproveDigitalRequest;
import io.gamedock.sdk.ads.providers.improvedigital.ImproveDigitalResponse;
import io.gamedock.sdk.ads.utils.error.ErrorCodes;
import io.gamedock.sdk.ads.utils.logging.LoggingUtilAds;
import io.gamedock.sdk.ads.utils.values.StaticValues;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;

/**
 * Logic class for handling all interstitial operations for the module.
 * All relevant callbacks are fired through the InterstitialAdListener.
 * The InterstitialAd is bound by two placement ids, one for video and one for static, and an ad provider (ex.:Improve Digital).
 */
public class InterstitialAd extends BaseAd {
    private InterstitialAdListener adListener;

    public InterstitialAd(AdProvider adProvider, String placementId, InterstitialAdListener adListener) {
        super(adProvider, new String[]{placementId});

        if (adListener == null) {
            throw new IllegalArgumentException("Ad listener cannot be null");
        }
        this.adListener = adListener;
    }

    public InterstitialAd(AdProvider adProvider, String vastPlacementId, String staticPlacementId, InterstitialAdListener adListener) {
        super(adProvider, new String[]{vastPlacementId, staticPlacementId});

        if (adListener == null) {
            throw new IllegalArgumentException("Ad listener cannot be null");
        }
        this.adListener = adListener;
    }

    /**
     * Method used to load an interstitial ad based on the supplied placement ids, one for video and one for static.
     * When making the load request, it will first make a request for the video placement and if no fill, it will fallback to the static one.
     */
    @Override
    public void load() {
        if (isLoading) {
            return;
        }

        if (!GamedockAds.getInstance().isInitialized()) {
            adListener.onAdFailedToLoad(ErrorCodes.NotInitialized);
            return;
        }

        final Context context = GamedockAds.getInstance().getContext();

        if (context == null) {
            adListener.onAdFailedToLoad(ErrorCodes.NullContext);
            return;
        }

        if (placementIds == null) {
            adListener.onAdFailedToLoad(ErrorCodes.NullPlacementId);
            return;
        }

        if (provider == AdProvider.IMPROVE_DIGITAL) {
            if (!GamedockAds.getInstance().isProviderInitialized(AdProvider.IMPROVE_DIGITAL)) {
                adListener.onAdFailedToLoad(ErrorCodes.NotInitialized);
                return;
            }
            ImproveDigitalRequest improveDigitalVastRequest = new ImproveDigitalRequest(context, placementIds[0], AdType.INTERSTITIAL, RequestType.VAST, new RequestListener() {
                @Override
                public void onSuccess(String responseVastJSON) {
                    ImproveDigitalResponse improveDigitalVastResponse = new ImproveDigitalResponse(responseVastJSON, AdType.INTERSTITIAL);

                    if (improveDigitalVastResponse.isError()) {
                        ImproveDigitalRequest improveDigitalStaticRequest = new ImproveDigitalRequest(context, placementIds[1], AdType.INTERSTITIAL, RequestType.STATIC, new RequestListener() {
                            @Override
                            public void onSuccess(String responseStaticJSON) {
                                ImproveDigitalResponse improveDigitalStaticResponse = new ImproveDigitalResponse(responseStaticJSON, AdType.INTERSTITIAL);
                                if (improveDigitalStaticResponse.isError()) {
                                    adListener.onAdFailedToLoad(improveDigitalStaticResponse.getError());
                                } else {
                                    adData = new AdData();

                                    adData.populateWithImproveDigitalResponse(improveDigitalStaticResponse);

                                    isLoading = false;
                                    isLoaded = true;

                                    adListener.onAdLoaded();
                                }
                            }

                            @Override
                            public void onFailure(ErrorCodes error) {
                                isLoading = false;
                                adListener.onAdFailedToLoad(error);
                            }
                        }, null);

                        improveDigitalStaticRequest.loadRequest().subscribeOn(Schedulers.computation())
                                .observeOn(AndroidSchedulers.mainThread())
                                .subscribe(improveDigitalStaticRequest.requestObserver);
                    } else {
                        adData = new AdData();

                        adData.populateWithImproveDigitalResponse(improveDigitalVastResponse);

                        isLoading = false;
                        isLoaded = true;

                        adListener.onAdLoaded();
                    }
                }

                @Override
                public void onFailure(ErrorCodes error) {
                    isLoading = false;
                    adListener.onAdFailedToLoad(error);
                }
            }, null);

            improveDigitalVastRequest.loadRequest().subscribeOn(Schedulers.computation())
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(improveDigitalVastRequest.requestObserver);
            isLoading = true;
        } else {
            adListener.onAdFailedToLoad(ErrorCodes.UnknownProvider);
        }
    }

    /**
     * Method used to show the loaded interstitial ad. If an ad has not been loaded an error callback will be fired.
     */
    @Override
    public void show() {
        if (!GamedockAds.getInstance().isInitialized()) {
            adListener.onAdFailedToDisplay(ErrorCodes.NotInitialized);
            return;
        }

        if (!isLoaded) {
            adListener.onAdFailedToDisplay(ErrorCodes.NotLoaded);
            return;
        }

        Context context = GamedockAds.getInstance().getContext();

        if (context == null) {
            adListener.onAdFailedToDisplay(ErrorCodes.NullContext);
            return;
        }

        LoggingUtilAds.d("Showing interstitial ad with data: " + new Gson().toJson(adData));

        GamedockAdsActivity.setInterstitialAd(this);

        Intent intent = new Intent(context, GamedockAdsActivity.class);
        intent.putExtra("adType", StaticValues.INTERSTITIAL);
        intent.putExtra("provider", getProvider().toString());
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

        context.startActivity(intent);
    }

    public void setAdListener(InterstitialAdListener adListener) {
        this.adListener = adListener;
    }

    public InterstitialAdListener getAdListener() {
        return adListener;
    }
}
