package io.gamedock.sdk.ads.core.banner;

import android.content.Context;

import io.gamedock.sdk.ads.utils.device.DensityUtil;
import io.gamedock.sdk.ads.utils.device.DeviceInfo;

/**
 * Class used to define the sizes for banner views.
 * If SMART_BANNER is selected, the size will be calculated based on the nearest accepted Width DP in combination with the height based on the device type.
 */
public class BannerAdSize {
    public static final BannerAdSize BANNER = new BannerAdSize(320, 50, "320x50_mb");
    public static final BannerAdSize FULL_BANNER = new BannerAdSize(468, 60, "468x60_as");
    public static final BannerAdSize LARGE_BANNER = new BannerAdSize(320, 100, "320x100_as");
    public static final BannerAdSize LEADERBOARD = new BannerAdSize(728, 90, "728x90_as");
    public static final BannerAdSize MEDIUM_RECTANGLE = new BannerAdSize(300, 250, "300x250_as");
    public static final BannerAdSize SMART_BANNER = new BannerAdSize(-1, -2, "smart_banner");

    private static int[] bannerAcceptedWidthDps = new int[]{300, 320, 468, 728};

    private int width;
    private int height;
    private String size;

    public BannerAdSize(int width, int height) {
        this(width, height, (width == -1 ? "FULL" : String.valueOf(width)) + "x" + (height == -2 ? "AUTO" : String.valueOf(height)) + "_as");
    }

    BannerAdSize(int width, int height, String size) {
        if (width < 0 && width != -1 && width != -3) {
            throw new IllegalArgumentException("Invalid width for AdSize: " + width);
        } else if (height < 0 && height != -2 && height != -4) {
            throw new IllegalArgumentException("Invalid height for AdSize: " + height);
        } else {
            this.width = width;
            this.height = height;
            this.size = size;
        }
    }

    /**
     * Method used to get the closest width DP for the size if the selected size is SMART_BANNER. The accepted DPs are: 300, 320, 468, 728.
     * @param target
     * @return
     */
    public int getClosestWidthDpToAccepted(int target) {
        int closestValue = bannerAcceptedWidthDps[0];
        int leastDistance = Math.abs(bannerAcceptedWidthDps[0] - target);
        for (int bannerAcceptedWidthDp : bannerAcceptedWidthDps) {
            int currentDistance = Math.abs(bannerAcceptedWidthDp - target);
            if (currentDistance < leastDistance && bannerAcceptedWidthDp < target) {
                closestValue = bannerAcceptedWidthDp;
                leastDistance = currentDistance;
            }
        }
        return closestValue;
    }

    /**
     * Method used to retrieve the width of the banner size in pixels.
     * If the size is SMART_BANNER, the closest width DP will be calculated and then converted to PX.
     * @param context
     * @return
     */
    public final int getWidthInPixels(Context context) {
        if (this.width == -1) {
            return DensityUtil.dpToPx(context, getClosestWidthDpToAccepted(DensityUtil.pxToDp(context, DeviceInfo.getDeviceWidth())));
        }

        return DensityUtil.dpToPx(context, this.width);
    }

    /**
     * Method used to retrieve the height of the banner size in pixels.
     * If the size is SMART_BANNER, the size will be calculated based on the density value of the screen (device type). The thresholds are at 400 and 720. The return value in these cases are: 32, 50, 90. The values are converted to PX.
     * @param context
     * @return
     */
    public final int getHeightInPixels(Context context) {
        if (this.height == -2) {
            int smartHeight;
            int deviceHeightDp = DensityUtil.pxToDp(context, DeviceInfo.getDeviceHeight());

            if (deviceHeightDp <= 400) {
                smartHeight = DensityUtil.dpToPx(context, 32);
            } else if (deviceHeightDp <= 720) {
                smartHeight = DensityUtil.dpToPx(context, 50);
            } else {
                smartHeight = DensityUtil.dpToPx(context, 90);
            }

            return smartHeight;
        }

        return DensityUtil.dpToPx(context, this.height);
    }

    /**
     * Method used to retrieve the width of the banner size in DP.
     * If the size is SMART_BANNER, the closest width DP will be calculated.
     * @param context
     * @return
     */
    public int getWidth(Context context) {
        if (this.width == -1) {
            return getClosestWidthDpToAccepted(DensityUtil.pxToDp(context, DeviceInfo.getDeviceWidth()));
        }

        return this.width;
    }

    /**
     * Method used to retrieve the height of the banner size in pixels.
     * If the size is SMART_BANNER, the size will be calculated based on the density value of the screen (device type). The thresholds are at 400 and 720. The return value in these cases are: 32, 50, 90.
     * @param context
     * @return
     */
    public int getHeight(Context context) {
        if (this.height == -2) {
            int smartHeight;
            int deviceHeightDp = DensityUtil.pxToDp(context, DeviceInfo.getDeviceHeight());

            if (deviceHeightDp <= 400) {
                smartHeight = 32;
            } else if (deviceHeightDp <= 720) {
                smartHeight = 50;
            } else {
                smartHeight = 90;
            }
            return smartHeight;
        }

        return this.height;
    }

    public String getSize() {
        return size;
    }
}
