package io.gamedock.sdk.ads;

import android.app.Activity;
import android.content.Context;
import android.util.Log;

import com.danikula.videocache.HttpProxyCacheServer;
import com.squareup.picasso.Picasso;

import java.util.ArrayList;

import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.ads.providers.improvedigital.ImproveDigitalProvider;
import io.gamedock.sdk.ads.providers.init.BaseAdProviderInit;
import io.gamedock.sdk.ads.providers.init.InitializationStatus;
import io.gamedock.sdk.ads.providers.init.OnGamedockAdsInitializationCompleted;
import io.gamedock.sdk.ads.utils.device.DeviceInfo;
import io.gamedock.sdk.ads.utils.file.FileUtils;
import io.gamedock.sdk.ads.utils.logging.LoggingUtilAds;
import io.gamedock.sdk.ads.utils.settings.Settings;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.annotations.NonNull;
import io.reactivex.disposables.Disposable;
import io.reactivex.schedulers.Schedulers;

public class GamedockAds {

    private static final Object lock = new Object();
    private static volatile GamedockAds mInstance;
    protected Context context;

    private boolean isInitialized;

    private Settings settings = new Settings();
    private ArrayList<InitializationStatus> initializationStatus = new ArrayList<>();

    private HttpProxyCacheServer videoProxyCache;

    private GamedockAds(Context context) {
        if (!(context instanceof Activity)) {
            throw new IllegalArgumentException("Context needs to be of Activity type");
        }
        this.context = context;
        Picasso picasso = new Picasso.Builder(context).build();
        Picasso.setSingletonInstance(picasso);
    }

    public static GamedockAds getInstance() {
        if (mInstance == null) {
            Log.e("GamedockAds","Ads module was not initialized correctly. Make sure you first call initialize(...) before using the instance.");
        }
        return mInstance;
    }

    /**
     * Method used to initialize the Gamedock Ads Module. Should be used before doing any other operations with the module.
     * @param context
     * @param adNetworksInit
     * @param settings
     * @param initializationCompleted
     */
    public static void initialize(Context context, ArrayList<BaseAdProviderInit> adNetworksInit, Settings settings, OnGamedockAdsInitializationCompleted initializationCompleted) {
        if (context == null) {
            LoggingUtilAds.e("Ads module cannot be initialized. Context is null.");
            return;
        }

        synchronized (lock) {
            if (mInstance == null) {
                mInstance = new GamedockAds(context);
                mInstance.initializeGamedockAds(settings, adNetworksInit, initializationCompleted);
            } else {
                LoggingUtilAds.w("Ads module instance already created. Initialize method was already called.");
            }
        }
    }

    /**
     * Method used internally to gather data relevant to the requests and operations of the module.
     * @param settings
     * @param adNetworksInit
     * @param initializationCompleted
     */
    private void initializeGamedockAds(Settings settings, ArrayList<BaseAdProviderInit> adNetworksInit, final OnGamedockAdsInitializationCompleted initializationCompleted) {
        if (settings != null) {
            this.settings.setWithUserConsent(settings.isWithUserConsent());
            this.settings.setGdprConsentString(settings.getGdprConsentString());
            this.settings.setTestModeEnabled(settings.isTestModeEnabled());
            this.settings.setDebugModeEnabled(settings.isDebugModeEnabled());
        }

        Observable.merge(DeviceInfo.retrieveRefererUrl(context),
                DeviceInfo.retrieveAdvertisingId(context),
                DeviceInfo.retrieveDeviceSizes(context),
                initializeAdNetworks(adNetworksInit))
                .subscribeOn(Schedulers.newThread())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Observer<String>() {
                    @Override
                    public void onSubscribe(@NonNull Disposable d) {
                    }

                    @Override
                    public void onNext(@NonNull String s) {
                    }

                    @Override
                    public void onError(@NonNull Throwable e) {
                    }

                    @Override
                    public void onComplete() {
                        DeviceInfo.retrievePackageName(context);
                        DeviceInfo.retrieveDeviceSpecs(context);

                        isInitialized = true;
                        LoggingUtilAds.d("Gamedock Ads Module Initialized");

                        if (initializationCompleted != null) {
                            initializationCompleted.onInitializationCompleted(initializationStatus);
                        }
                    }
                });
    }

    /**
     * Method used to initialized the different providers present in the module.
     * @param adNetworksInit
     * @return
     */
    private Observable<String> initializeAdNetworks(final ArrayList<BaseAdProviderInit> adNetworksInit) {
        return Observable.create(new ObservableOnSubscribe<String>() {
            @Override
            public void subscribe(ObservableEmitter<String> observableEmitter) throws Exception {
                try {
                    for (int i = 0; i < adNetworksInit.size(); i++) {
                        if (adNetworksInit.get(i) instanceof ImproveDigitalProvider.Init) {
                            InitializationStatus improveDigitalStatus = ImproveDigitalProvider.initializeImproveDigital((ImproveDigitalProvider.Init) adNetworksInit.get(i));
                            GamedockAds.getInstance().initializationStatus.add(improveDigitalStatus);
                        }
                    }

                    LoggingUtilAds.d("Ad Networks initialized");
                    observableEmitter.onComplete();
                } catch (Exception e) {
                    observableEmitter.onError(e);
                }
            }
        });
    }

    /**
     * Method used to check if a specific provider (ex.:Improve Digital) has been initialized.
     * @param provider
     * @return
     */
    public boolean isProviderInitialized(AdProvider provider) {
        if (!isInitialized) {
            return false;
        }

        boolean providerInitialized = false;

        for (InitializationStatus status : initializationStatus) {
            if (status.getAdProvider() == provider) {
                providerInitialized = true;
                break;
            }
        }

        return providerInitialized;
    }

    public HttpProxyCacheServer getProxy() {
        return videoProxyCache == null ? (videoProxyCache = newVideoProxyCache()) : videoProxyCache;
    }

    private HttpProxyCacheServer newVideoProxyCache() {
        return new HttpProxyCacheServer.Builder(context)
                .cacheDirectory(FileUtils.getVideoCacheDir(context))
                .build();
    }

    public void setSettings(Settings settings) {
        this.settings = settings;
    }

    public Settings getSettings() {
        return settings;
    }

    public boolean isInitialized() {
        return isInitialized;
    }

    public Context getContext() {
        return context;
    }

    public ArrayList<InitializationStatus> getInitializationStatus() {
        return initializationStatus;
    }
}
