package io.gamedock.sdk.ads.utils.connectivity;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.telephony.TelephonyManager;

/**
 * Class used to gather all information related to the network connection.
 */
public class ConnectivityInfo {
    private static final String CONN_TYPE_WIRELESS = "wifi";
    private static final String CONN_TYPE_EDGE = "edge";
    private static final String CONN_TYPE_3G = "3g";
    private static final String CONN_TYPE_4G = "4g";

    private static final int NO_CONNECTION = 100;
    private static final int TYPE_MOBILE = 101;
    private static final int TYPE_WIFI = 102;

    /**
     * Method used to retrieve the connection speed of the device.
     * @param context
     * @return
     */
    public static String retrieveConnectionSpeed(Context context) {
        int connectionType = getConnectionType(context);

        if (connectionType == TYPE_WIFI) {
            return CONN_TYPE_WIRELESS;
        } else if (connectionType == TYPE_MOBILE) {
            try {
                TelephonyManager telephonyManager = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
                int networkType = telephonyManager.getNetworkType();
                switch (networkType) {
                    case TelephonyManager.NETWORK_TYPE_GPRS:
                    case TelephonyManager.NETWORK_TYPE_EDGE:
                    case TelephonyManager.NETWORK_TYPE_CDMA:
                    case TelephonyManager.NETWORK_TYPE_1xRTT:
                    case TelephonyManager.NETWORK_TYPE_IDEN:
                        return CONN_TYPE_EDGE;
                    case TelephonyManager.NETWORK_TYPE_UMTS:
                    case TelephonyManager.NETWORK_TYPE_EVDO_0:
                    case TelephonyManager.NETWORK_TYPE_EVDO_A:
                    case TelephonyManager.NETWORK_TYPE_HSDPA:
                    case TelephonyManager.NETWORK_TYPE_HSUPA:
                    case TelephonyManager.NETWORK_TYPE_HSPA:
                    case TelephonyManager.NETWORK_TYPE_EVDO_B:
                    case TelephonyManager.NETWORK_TYPE_EHRPD:
                    case TelephonyManager.NETWORK_TYPE_HSPAP:
                        return CONN_TYPE_3G;
                    case TelephonyManager.NETWORK_TYPE_LTE:
                        return CONN_TYPE_4G;
                    default:
                        break;
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        return null;
    }

    /**
     * Method to get the connection type from the device.
     * @param context
     * @return
     */
    private static int getConnectionType(Context context) {
        try {
            NetworkInfo info = getNetworkInfo(context);
            if (info != null && info.isConnected()) {
                switch (info.getType()) {
                    case ConnectivityManager.TYPE_MOBILE:
                        return TYPE_MOBILE;
                    case ConnectivityManager.TYPE_WIFI:
                        return TYPE_WIFI;
                    default:
                        break;
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return NO_CONNECTION;
    }

    private static NetworkInfo getNetworkInfo(Context context) {
        try {
            ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            if (cm != null) {
                return cm.getActiveNetworkInfo();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return null;
    }

    /**
     * Method used to get the carrier for the device if possible.
     * @param context
     * @return
     */
    public static String retrieveCarrier(Context context) {
        try {
            TelephonyManager telephonyManager = ((TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE));
            return telephonyManager.getNetworkOperatorName();
        } catch (Exception e) {
            e.printStackTrace();
        }

        return null;
    }

    /**
     * Method used to check if the internet connection is available.
     * @param context
     * @return
     */
    public static boolean isInternetAvailable(Context context) {
        try {
            ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo netInfo = null;
            if (cm != null) {
                netInfo = cm.getActiveNetworkInfo();
            }
            return netInfo != null && netInfo.isConnectedOrConnecting();
        } catch (NullPointerException e) {
            return false;
        }
    }


}
