package io.gamedock.sdk.ads.core.base.web;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Intent;
import android.graphics.Bitmap;
import android.net.Uri;
import android.webkit.WebResourceError;
import android.webkit.WebResourceRequest;
import android.webkit.WebResourceResponse;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import io.gamedock.sdk.ads.core.banner.BannerAd;
import io.gamedock.sdk.ads.core.interstitial.InterstitialAd;
import io.gamedock.sdk.ads.core.tracking.TrackingRequest;
import io.gamedock.sdk.ads.utils.error.ErrorCodes;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;

/**
 * Class used to add additional callbacks and logic to the WebView through the WebViewClient.
 */
public class AdWebViewClient extends WebViewClient {

    private Activity activity;
    private boolean isBanner;
    private InterstitialAd interstitialAd;
    private BannerAd bannerAd;

    public AdWebViewClient(Activity activity, boolean isBanner, InterstitialAd interstitialAd, BannerAd bannerAd) {
        this.activity = activity;
        this.isBanner = isBanner;
        this.interstitialAd = interstitialAd;
        this.bannerAd = bannerAd;
    }

    @Override
    public void onPageStarted(WebView view, String url, Bitmap favicon) {
        super.onPageStarted(view, url, favicon);
    }

    @Override
    public void onPageFinished(WebView view, String url) {
        super.onPageFinished(view, url);
        if (isBanner) {
            bannerAd.setLoading(false);
            bannerAd.setLoaded(true);

            bannerAd.getAdListener().onAdLoaded();
        } else {
            interstitialAd.getAdListener().onAdDisplayed();

            if (interstitialAd.getAdData().getBurl() != null) {
                TrackingRequest trackingRequest = new TrackingRequest(activity, interstitialAd.getAdData().getBurl());
                trackingRequest.loadRequest().subscribeOn(Schedulers.computation())
                        .observeOn(AndroidSchedulers.mainThread())
                        .subscribe(trackingRequest.requestObserver);
            }
        }
    }

    @Override
    public void onLoadResource(WebView view, String url) {
        super.onLoadResource(view, url);
    }
    @Override
    public boolean shouldOverrideUrlLoading(WebView view, String url) {
        if (isBanner) {
            bannerAd.getAdListener().onAdClicked();
        } else {
            interstitialAd.getAdListener().onAdClicked();
        }

        try {
            Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));
            activity.startActivity(browserIntent);
        } catch (NullPointerException | ActivityNotFoundException e) {
            e.printStackTrace();
        }

        return true;
    }

    @Override
    public void onReceivedError(WebView view, WebResourceRequest request, WebResourceError error) {
        super.onReceivedError(view, request, error);

        if (isBanner) {
            bannerAd.setLoading(false);
            bannerAd.getAdListener().onAdFailedToDisplay(ErrorCodes.PlayerError);
        } else {
            interstitialAd.getAdListener().onAdFailedToDisplay(ErrorCodes.PlayerError);
            if (activity != null) {
                activity.finish();
            }
        }
    }

    @Override
    public void onReceivedHttpError(WebView view, WebResourceRequest request, WebResourceResponse errorResponse) {
        super.onReceivedHttpError(view, request, errorResponse);

        if (isBanner) {
            bannerAd.setLoading(false);
            bannerAd.getAdListener().onAdFailedToDisplay(ErrorCodes.PlayerError);
        } else {
            interstitialAd.getAdListener().onAdFailedToDisplay(ErrorCodes.PlayerError);
            if (activity != null) {
                activity.finish();
            }
        }
    }
}
