package io.gamedock.sdk.ads.core.banner;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.squareup.picasso.Callback;
import com.squareup.picasso.Picasso;

import io.gamedock.sdk.ads.GamedockAds;
import io.gamedock.sdk.ads.R;
import io.gamedock.sdk.ads.core.base.web.AdWebChromeClient;
import io.gamedock.sdk.ads.core.base.web.AdWebView;
import io.gamedock.sdk.ads.core.base.web.AdWebViewClient;
import io.gamedock.sdk.ads.core.base.web.AdWebViewPreparer;
import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.ads.providers.improvedigital.ImproveDigitalProvider;
import io.gamedock.sdk.ads.utils.device.DensityUtil;
import io.gamedock.sdk.ads.utils.error.ErrorCodes;
import io.gamedock.sdk.ads.utils.values.StaticValues;

/**
 * Internal class that handles the Banner View.
 * The View can be populated with an Image or a WebView.
 */
public class BannerAdView extends ViewGroup {
    protected BannerAdSize adSize;
    protected BannerAdPosition adPosition;
    protected AdWebView adWebView;
    protected ImageView staticImageView;
    private RelativeLayout privacyPolicy;
    private ImageView privacyPolicyImage;
    private TextView privacyPolicyText;
    protected BannerAdListener adListener;
    private Context context;
    private RelativeLayout viewLayout;

    public BannerAdView(Context context) {
        super(context);
        this.context = context;
    }

    public BannerAdView(Context context, AttributeSet attrs) {
        super(context, attrs);
    }

    public BannerAdView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
    }

    /**
     * Method used to load the banner ad view based on the type, size and position.
     * @param bannerAd
     * @param adSize
     * @param adPosition
     * @param adListener
     */
    protected void loadAd(BannerAd bannerAd, BannerAdSize adSize, BannerAdPosition adPosition, BannerAdListener adListener) {
        this.adSize = adSize;
        this.adPosition = adPosition;
        this.adListener = adListener;

        int gravity = this.adPosition == BannerAdPosition.TOP ? Gravity.TOP : Gravity.BOTTOM;

        FrameLayout.LayoutParams viewLayoutParams = new FrameLayout.LayoutParams(adSize.getWidthInPixels(context), adSize.getHeightInPixels(context), Gravity.CENTER_HORIZONTAL | gravity);

        this.setLayoutParams(viewLayoutParams);

        if (viewLayout == null) {
            viewLayout = new RelativeLayout(context);
            viewLayout.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));
            this.addView(viewLayout);
        } else {
            try {
                viewLayout.removeAllViews();
            } catch (NullPointerException ignored) {
            }
        }

        switch (bannerAd.getAdData().getType()) {
            case StaticValues.IMAGE:
                loadStaticImage(bannerAd, viewLayout);
                break;
            case StaticValues.HTML:
                loadHTML(bannerAd, viewLayout);
                break;
        }

        configurePrivacyPolicy(bannerAd.getProvider(), viewLayout);
    }

    /**
     * Method used to load a static image into an ImageView. Once the ImageView is loaded, it is attached to the overall BannerView layout.
     * @param bannerAd
     * @param layout
     */
    private void loadStaticImage(final BannerAd bannerAd, RelativeLayout layout) {
        staticImageView = new ImageView(context);

        staticImageView.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));

        layout.addView(staticImageView);

        staticImageView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                try {
                    adListener.onAdClicked();
                    Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(bannerAd.getAdData().getClickUrl()));
                    context.startActivity(browserIntent);
                } catch (NullPointerException | ActivityNotFoundException e) {
                    e.printStackTrace();
                }
            }
        });

        Picasso.get().load(Uri.parse(bannerAd.getAdData().getAdm())).into(staticImageView, new Callback() {
            @Override
            public void onSuccess() {
                bannerAd.setLoading(false);
                bannerAd.setLoaded(true);

                adListener.onAdLoaded();
            }

            @Override
            public void onError(Exception e) {
                bannerAd.setLoading(false);
                adListener.onAdFailedToLoad(ErrorCodes.AssetDownload);
            }
        });
    }

    /**
     * Method used to load a static image into a WebView. Once the WebView is loaded, it is attached to the overall BannerView layout.
     * @param bannerAd
     * @param layout
     */
    private void loadHTML(BannerAd bannerAd, RelativeLayout layout) {
        adWebView = new AdWebView(context, this);

        adWebView.setLayoutParams(new LayoutParams(LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT));
        adWebView.requestLayout();
        layout.addView(adWebView);

        String content = AdWebViewPreparer.prepareWebViewUrl(bannerAd.getAdData().getAdm());
        adWebView.setWebViewClient(new AdWebViewClient((Activity) context, true, null, bannerAd));
        adWebView.setWebChromeClient(new AdWebChromeClient());
        adWebView.setHorizontalScrollBarEnabled(false);
        adWebView.setVerticalScrollBarEnabled(false);
        adWebView.loadDataWithBaseURL(AdWebViewPreparer.HTML_BASE_URL, content, "text/html", "UTF-8", null);
    }

    /**
     * Method used to add the privacy policy button from the Banner View.
     * @param provider
     * @param layout
     */
    private void configurePrivacyPolicy(AdProvider provider, RelativeLayout layout) {
        RelativeLayout.LayoutParams viewLayoutParams = new RelativeLayout.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
        viewLayoutParams.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
        viewLayoutParams.addRule(RelativeLayout.ALIGN_PARENT_LEFT);
        viewLayoutParams.leftMargin = DensityUtil.dpToPx(context, 5);
        viewLayoutParams.bottomMargin = DensityUtil.dpToPx(context, 5);

        privacyPolicy = new RelativeLayout(context);
        privacyPolicy.setLayoutParams(viewLayoutParams);

        int imageViewId = View.generateViewId();
        RelativeLayout.LayoutParams imageLayoutParams = new RelativeLayout.LayoutParams(DensityUtil.dpToPx(context, 16), DensityUtil.dpToPx(context, 16));

        privacyPolicyImage = new ImageView(context);
        privacyPolicyImage.setId(imageViewId);
        privacyPolicyImage.setLayoutParams(imageLayoutParams);

        RelativeLayout.LayoutParams textLayoutParams = new RelativeLayout.LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
        textLayoutParams.addRule(RelativeLayout.END_OF, imageViewId);
        textLayoutParams.leftMargin = DensityUtil.dpToPx(context, 5);

        privacyPolicyText = new TextView(context);
        privacyPolicyText.setTextSize(TypedValue.COMPLEX_UNIT_SP, 10);
        privacyPolicyText.setTextColor(Color.WHITE);
        privacyPolicyText.setShadowLayer(2, (float) 1.5, (float) 1.3, Color.BLACK);
        privacyPolicyText.setGravity(Gravity.CENTER_VERTICAL);

        privacyPolicyText.setLayoutParams(textLayoutParams);

        String text = "";
        String url;
        Drawable logo;

        Activity activity = (Activity) GamedockAds.getInstance().getContext();

        switch (provider) {
            case IMPROVE_DIGITAL:
                //text = "Ads by Improve Digital";
                url = ImproveDigitalProvider.PRIVACY_POLICY_URL;
                logo = activity.getResources().getDrawable(R.drawable.ic_improve_digital);
                break;
            case GAMEDOCK:
            default:
                //text = "Ads by Gamedock";
                url = "https://azerion.com/business/mobile-apps-privacy-policy.html";
                logo = activity.getResources().getDrawable(R.drawable.ic_close);
                break;
        }

        privacyPolicyText.setText(text);
        privacyPolicyImage.setBackground(logo);
        final String finalUrl = url;
        privacyPolicy.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                try {
                    Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(finalUrl));
                    context.startActivity(browserIntent);
                } catch (NullPointerException | ActivityNotFoundException e) {
                    e.printStackTrace();
                }
            }
        });

        privacyPolicy.addView(privacyPolicyImage);
        privacyPolicy.addView(privacyPolicyText);

        layout.addView(privacyPolicy);
        privacyPolicy.bringToFront();
    }

    @Override
    protected void onLayout(boolean changed, int l, int t, int r, int b) {
        View view;
        if ((view = this.getChildAt(0)) != null && view.getVisibility() != GONE) {
            int width = view.getMeasuredWidth();
            int height = view.getMeasuredHeight();
            int left = (r - l - width) / 2;
            int top = (b - t - height) / 2;
            view.layout(left, top, left + width, top + height);
        }
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        int width = 0;
        int height = 0;
        View view;
        if ((view = this.getChildAt(0)) != null && view.getVisibility() != GONE) {
            this.measureChild(view, widthMeasureSpec, heightMeasureSpec);
            width = view.getMeasuredWidth();
            height = view.getMeasuredHeight();
        } else {
            BannerAdSize adSize = null;

            try {
                adSize = this.adSize;
            } catch (NullPointerException e) {
                e.printStackTrace();
            }

            if (adSize != null) {
                width = adSize.getWidthInPixels(context);
                height = adSize.getHeightInPixels(context);
            }
        }

        width = Math.max(width, this.getSuggestedMinimumWidth());
        height = Math.max(height, this.getSuggestedMinimumHeight());
        this.setMeasuredDimension(View.resolveSize(width, widthMeasureSpec), View.resolveSize(height, heightMeasureSpec));
    }

    public void setAdSize(BannerAdSize adSize) {
        this.adSize = adSize;
    }
}
