package io.gamedock.sdk.ads.core.banner;

import android.app.Activity;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.view.View;
import android.view.ViewGroup;

import com.google.gson.Gson;

import io.gamedock.sdk.ads.GamedockAds;
import io.gamedock.sdk.ads.core.base.AdData;
import io.gamedock.sdk.ads.core.base.BaseAd;
import io.gamedock.sdk.ads.core.request.AdType;
import io.gamedock.sdk.ads.core.request.RequestListener;
import io.gamedock.sdk.ads.core.request.RequestType;
import io.gamedock.sdk.ads.core.tracking.TrackingRequest;
import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.ads.providers.improvedigital.ImproveDigitalRequest;
import io.gamedock.sdk.ads.providers.improvedigital.ImproveDigitalResponse;
import io.gamedock.sdk.ads.utils.error.ErrorCodes;
import io.gamedock.sdk.ads.utils.logging.LoggingUtilAds;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;

/**
 * Logic class for handling all banner operations for the module.
 * All relevant callbacks are fired through the BannerAdListener.
 * The BannerAd is bound by a placement id and an ad provider (ex.:Improve Digital).
 */
public class BannerAd extends BaseAd {
    private BannerAdListener adListener;
    private BannerAdView adView;
    private BannerAdSize adSize = BannerAdSize.SMART_BANNER;
    private BannerAdPosition adPosition = BannerAdPosition.BOTTOM;
    private boolean isShowing;
    private boolean wasDestroyed;
    private boolean burlTracked;

    private int bannerAdViewId = View.generateViewId();

    private int refreshRate = 60000;
    private Handler refreshHandler;
    private Runnable refreshRunnable;

    public BannerAd(AdProvider network, String placementId, BannerAdListener adListener) {
        super(network, new String[]{placementId});
        Context context = GamedockAds.getInstance().getContext();
        adView = new BannerAdView(context);

        if (adListener == null) {
            throw new IllegalArgumentException("Ad listener cannot be null");
        }
        this.adListener = adListener;

        refreshHandler = new Handler(Looper.getMainLooper());
        refreshRunnable = new Runnable() {
            @Override
            public void run() {
                refreshAd();

                if (refreshHandler != null) {
                    refreshHandler.postDelayed(refreshRunnable, refreshRate);
                }
            }
        };

        refreshHandler.postDelayed(refreshRunnable, refreshRate);
    }

    /**
     * Method used to load a banner ad based on the supplied placement id.
     */
    @Override
    public void load() {
        if (isLoading) {
            return;
        }

        if (!GamedockAds.getInstance().isInitialized()) {
            adListener.onAdFailedToLoad(ErrorCodes.NotInitialized);
            return;
        }

        Context context = GamedockAds.getInstance().getContext();

        if (context == null) {
            adListener.onAdFailedToLoad(ErrorCodes.NullContext);
            return;
        }

        if (wasDestroyed) {
            adListener.onAdFailedToLoad(ErrorCodes.ViewDestroyed);
            return;
        }

        if (placementIds == null || placementIds[0] == null || placementIds[0].isEmpty()) {
            adListener.onAdFailedToLoad(ErrorCodes.NullPlacementId);
            return;
        }

        if (provider == AdProvider.IMPROVE_DIGITAL) {
            if (!GamedockAds.getInstance().isProviderInitialized(AdProvider.IMPROVE_DIGITAL)) {
                adListener.onAdFailedToLoad(ErrorCodes.NotInitialized);
                return;
            }

            final BannerAd bannerAd = this;

            ImproveDigitalRequest improveDigitalRequest = new ImproveDigitalRequest(context, placementIds[0], AdType.BANNER, RequestType.STATIC, new RequestListener() {
                @Override
                public void onSuccess(String responseJSON) {
                    ImproveDigitalResponse improveDigitalResponse = new ImproveDigitalResponse(responseJSON, AdType.BANNER);

                    if (improveDigitalResponse.isError()) {
                        adListener.onAdFailedToLoad(improveDigitalResponse.getError());
                    } else {
                        adData = new AdData();

                        adData.populateWithImproveDigitalResponse(improveDigitalResponse);

                        adView.loadAd(bannerAd, adSize, adPosition, adListener);
                        wasDestroyed = false;
                    }
                }

                @Override
                public void onFailure(ErrorCodes error) {
                    isLoading = false;
                    adListener.onAdFailedToLoad(error);
                }
            }, adSize);

            improveDigitalRequest.loadRequest().subscribeOn(Schedulers.computation())
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(improveDigitalRequest.requestObserver);
            isLoading = true;
        } else {
            adListener.onAdFailedToLoad(ErrorCodes.UnknownProvider);
        }
    }

    /**
     * Method used to show the loaded banner ad. If an ad has not been loaded an error callback will be fired.
     */
    @Override
    public void show() {
        if (isShowing) {
            return;
        }

        if (wasDestroyed || adView == null) {
            adListener.onAdFailedToLoad(ErrorCodes.ViewDestroyed);
            return;
        }

        if (!GamedockAds.getInstance().isInitialized()) {
            adListener.onAdFailedToLoad(ErrorCodes.NotInitialized);
            return;
        }

        if (!isLoaded) {
            adListener.onAdFailedToDisplay(ErrorCodes.NotLoaded);
            return;
        }

        Context context = GamedockAds.getInstance().getContext();

        if (context == null) {
            adListener.onAdFailedToDisplay(ErrorCodes.NullContext);
            return;
        }

        LoggingUtilAds.d("Showing banner ad with data: " + new Gson().toJson(adData));

        final Activity activity = (Activity) context;

        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                ViewGroup viewGroup = (ViewGroup) activity.getWindow().getDecorView().getRootView();

                if (viewGroup == null) {
                    adListener.onAdFailedToDisplay(ErrorCodes.PlayerError);
                    return;
                }

                if (activity.findViewById(bannerAdViewId) != null || (adView.getParent() != null && adView.getParent() == viewGroup)) {
                    setVisibility(View.VISIBLE);
                } else {
                    adView.setId(bannerAdViewId);

                    viewGroup.addView(adView);
                    adView.bringToFront();
                    setVisibility(View.VISIBLE);
                }
                isShowing = true;
                adListener.onAdDisplayed();

                if (!burlTracked) {
                    if (adData.getBurl() != null) {
                        TrackingRequest trackingRequest = new TrackingRequest(activity, adData.getBurl());
                        trackingRequest.loadRequest().subscribeOn(Schedulers.computation())
                                .observeOn(AndroidSchedulers.mainThread())
                                .subscribe(trackingRequest.requestObserver);

                        burlTracked = true;
                    }
                }
            }
        });
    }

    /**
     * Method used to hide the displaying banner. If a banner is not showing or it has already been destroyed an error callback will fire.
     */
    public void hide() {
        if (!isShowing) {
            return;
        }

        if (wasDestroyed) {
            adListener.onAdFailedToLoad(ErrorCodes.ViewDestroyed);
            return;
        }

        Context context = GamedockAds.getInstance().getContext();

        final Activity activity = (Activity) context;

        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                if (activity.findViewById(bannerAdViewId) != null) {
                    activity.findViewById(bannerAdViewId).setVisibility(View.GONE);
                }

                adView.setVisibility(View.GONE);

                isShowing = false;

                adListener.onAdHidden();
            }
        });
    }

    /**
     * Method used to completely remove the banner view from the application.
     */
    public void destroy() {
        hide();

        Context context = GamedockAds.getInstance().getContext();

        final Activity activity = (Activity) context;

        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                try {
                    adView.removeAllViews();

                    ViewGroup viewGroup = (ViewGroup) adView.getParent();
                    if (viewGroup != null) {
                        viewGroup.removeView(adView);
                    }

                    adView = null;
                    refreshHandler.removeCallbacks(refreshRunnable);
                } catch (Exception e) {
                    e.printStackTrace();
                }

                wasDestroyed = true;
                isShowing = false;
                isLoaded = false;
            }
        });
    }

    /**
     * Method used internally to refresh the banner ad view with a new creative. This happens regularly at a 60 seconds interval if at the point of refresh the banner ad is visible.
     */
    private void refreshAd() {
        if (!isShowing) {
            return;
        }

        final Context context = GamedockAds.getInstance().getContext();

        if (context == null) {
            adListener.onAdFailedToDisplay(ErrorCodes.NullContext);
            return;
        }

        LoggingUtilAds.d("Refreshing banner ad");

        if (provider == AdProvider.IMPROVE_DIGITAL) {
            final BannerAd bannerAd = this;

            ImproveDigitalRequest improveDigitalRequest = new ImproveDigitalRequest(context, placementIds[0], AdType.BANNER, RequestType.STATIC, new RequestListener() {
                @Override
                public void onSuccess(String responseJSON) {
                    ImproveDigitalResponse improveDigitalResponse = new ImproveDigitalResponse(responseJSON, AdType.BANNER);

                    if (improveDigitalResponse.isError()) {
                        adListener.onAdFailedToLoad(improveDigitalResponse.getError());
                    } else {
                        adData = new AdData();

                        adData.populateWithImproveDigitalResponse(improveDigitalResponse);

                        if (!wasDestroyed) {
                            adView.loadAd(bannerAd, adSize, adPosition, adListener);

                            LoggingUtilAds.d("Showing banner ad with data: " + new Gson().toJson(adData));

                            if (adData.getBurl() != null) {
                                TrackingRequest trackingRequest = new TrackingRequest(context, adData.getBurl());
                                trackingRequest.loadRequest().subscribeOn(Schedulers.computation())
                                        .observeOn(AndroidSchedulers.mainThread())
                                        .subscribe(trackingRequest.requestObserver);

                                burlTracked = true;
                            }
                        }
                    }
                }

                @Override
                public void onFailure(ErrorCodes error) {
                    isLoading = false;
                    adListener.onAdFailedToLoad(error);
                }
            }, adSize);

            improveDigitalRequest.loadRequest().subscribeOn(Schedulers.computation())
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(improveDigitalRequest.requestObserver);
            isLoading = true;
        } else {

        }
    }

    public void setAdSize(BannerAdSize adSize) {
        this.adSize = adSize;
    }

    public void setAdPosition(BannerAdPosition adPosition) {
        this.adPosition = adPosition;
    }

    public void setLayoutParameters(ViewGroup.LayoutParams layoutParameters) {
        this.adView.setLayoutParams(layoutParameters);
    }

    public void setVisibility(int visibility) {
        adView.setVisibility(visibility);
    }

    public BannerAdListener getAdListener() {
        return adListener;
    }

    public void setAdListener(BannerAdListener adListener) {
        this.adListener = adListener;
    }

    public boolean isShowing() {
        return isShowing;
    }
}
