package io.gamedock.sdk.ads.core.ui;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.view.View;
import android.widget.Button;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.Nullable;

import com.danikula.videocache.CacheListener;
import com.danikula.videocache.HttpProxyCacheServer;
import com.google.ads.interactivemedia.v3.api.AdEvent;
import com.google.android.exoplayer2.ExoPlaybackException;
import com.google.android.exoplayer2.ExoPlayer;
import com.google.android.exoplayer2.SimpleExoPlayer;
import com.google.android.exoplayer2.analytics.AnalyticsListener;
import com.google.android.exoplayer2.ext.ima.ImaAdsLoader;
import com.google.android.exoplayer2.source.LoadEventInfo;
import com.google.android.exoplayer2.source.MediaLoadData;
import com.google.android.exoplayer2.source.MediaSource;
import com.google.android.exoplayer2.source.MediaSourceEventListener;
import com.google.android.exoplayer2.source.ProgressiveMediaSource;
import com.google.android.exoplayer2.source.ads.AdsMediaSource;
import com.google.android.exoplayer2.ui.PlayerView;
import com.google.android.exoplayer2.upstream.DataSource;
import com.google.android.exoplayer2.upstream.DefaultDataSourceFactory;
import com.google.android.exoplayer2.util.Util;
import com.squareup.picasso.Callback;
import com.squareup.picasso.Picasso;

import java.io.File;
import java.io.IOException;

import io.gamedock.sdk.ads.GamedockAds;
import io.gamedock.sdk.ads.R;
import io.gamedock.sdk.ads.core.base.AdData;
import io.gamedock.sdk.ads.core.base.BaseAdListener;
import io.gamedock.sdk.ads.core.base.web.AdWebChromeClient;
import io.gamedock.sdk.ads.core.base.web.AdWebView;
import io.gamedock.sdk.ads.core.base.web.AdWebViewClient;
import io.gamedock.sdk.ads.core.base.web.AdWebViewPreparer;
import io.gamedock.sdk.ads.core.interstitial.InterstitialAd;
import io.gamedock.sdk.ads.core.rewarded.RewardedVideoAd;
import io.gamedock.sdk.ads.core.rewarded.RewardedVideoAdListener;
import io.gamedock.sdk.ads.core.tracking.TrackingRequest;
import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.ads.providers.improvedigital.ImproveDigitalProvider;
import io.gamedock.sdk.ads.utils.error.ErrorCodes;
import io.gamedock.sdk.ads.utils.logging.LoggingUtilAds;
import io.gamedock.sdk.ads.utils.values.StaticValues;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;

/**
 * Activity class used for displaying interstitials and rewarded videos ads from the module.
 */
public class GamedockAdsActivity extends Activity {

    private GamedockAdsActivity activity;

    private static InterstitialAd interstitialAd;
    private static RewardedVideoAd rewardedVideoAd;

    private String adType;

    private Handler handler = new Handler(Looper.getMainLooper());
    private Runnable runnable;

    private RelativeLayout staticImageLayout;
    private ImageView staticImageView;
    private ImageButton staticImageCloseButton;

    private RelativeLayout htmlLayout;
    private AdWebView adWebView;
    private ImageButton htmlCloseButton;

    private RelativeLayout vastLayout;
    private PlayerView vastPlayer;
    private SimpleExoPlayer simpleVastPlayer;
    private ImaAdsLoader adsLoader;
    private String vastContent;
    private boolean shouldRenderVAST;

    private RelativeLayout videoLayout;
    private PlayerView videoPlayer;
    private SimpleExoPlayer simpleVideoPlayer;
    private Button videoPlayerClickButton;
    private ImageButton videoCloseButton;
    private CacheListener videoPlayerCacheListener;
    private boolean shouldRenderVideo;
    private boolean shouldGiveReward;

    private String burl;

    private RelativeLayout privacyPolicy;
    private ImageView privacyPolicyImage;
    private TextView privacyPolicyText;
    private AdProvider provider;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_gamedock_ads);

        try {
            overridePendingTransition(android.R.anim.slide_in_left, android.R.anim.slide_out_right);
        } catch (Exception ignored) {
        }

        activity = this;

        Bundle extras = getIntent().getExtras();

        if (!verifyActivityParameters(extras)) {
            LoggingUtilAds.e("Activity parameters are null when showing ad");
            activity.finish();
            return;
        }

        staticImageLayout = findViewById(R.id.staticImageLayout);
        staticImageView = findViewById(R.id.staticImageView);
        staticImageCloseButton = findViewById(R.id.staticImageCloseButton);

        htmlLayout = findViewById(R.id.htmlLayout);
        adWebView = findViewById(R.id.htmlWebView);
        htmlCloseButton = findViewById(R.id.htmlCloseButton);

        vastLayout = findViewById(R.id.vastLayout);
        vastPlayer = findViewById(R.id.vastPlayer);

        videoLayout = findViewById(R.id.videoLayout);
        videoPlayer = findViewById(R.id.videoPlayer);
        videoPlayerClickButton = findViewById(R.id.videoPlayerClickButton);
        videoCloseButton = findViewById(R.id.videoCloseButton);

        privacyPolicy = findViewById(R.id.privacyPolicy);
        privacyPolicyImage = findViewById(R.id.privacyPolicyImage);
        privacyPolicyText = findViewById(R.id.privacyPolicyText);

        configurePrivacyPolicy();

        switch (adType) {
            case StaticValues.INTERSTITIAL:
                if (interstitialAd == null) {
                    LoggingUtilAds.e("Interstitial ad is null when showing ad");
                    activity.finish();
                    return;
                }

                final AdData interstitialAdData = interstitialAd.getAdData();
                burl = interstitialAdData.getBurl();
                switch (interstitialAdData.getType()) {
                    case StaticValues.IMAGE:
                        renderStaticImage(interstitialAdData);
                        break;
                    case StaticValues.HTML:
                        renderHTML(interstitialAdData);
                        break;
                    case StaticValues.VIDEO:
                        shouldRenderVideo = true;
                        renderVideo(interstitialAdData);
                        break;
                    case StaticValues.VAST:
                        vastContent = interstitialAdData.getAdm();
                        shouldRenderVAST = true;
                        break;
                    default:
                        interstitialAd.getAdListener().onAdFailedToLoad(ErrorCodes.InvalidResponse);
                        activity.finish();
                        return;
                }
                break;
            case StaticValues.REWARDED_VIDEO:
                if (rewardedVideoAd == null) {
                    LoggingUtilAds.e("Rewarded Video ad is null when showing ad");
                    activity.finish();
                    return;
                }

                final AdData rewardedVideoAdData = rewardedVideoAd.getAdData();
                burl = rewardedVideoAdData.getBurl();

                switch (rewardedVideoAdData.getType()) {
                    case StaticValues.VIDEO:
                        shouldRenderVideo = true;
                        renderVideo(rewardedVideoAdData);
                        break;
                    case StaticValues.VAST:
                        vastContent = rewardedVideoAdData.getAdm();
                        shouldRenderVAST = true;
                        break;
                    default:
                        rewardedVideoAd.getAdListener().onAdFailedToLoad(ErrorCodes.InvalidResponse);
                        activity.finish();
                        return;
                }
                break;
        }
    }

    /**
     * Method used to render a static image ad using the Picasso library.
     * @param adData
     */
    private void renderStaticImage(final AdData adData) {
        staticImageLayout.setVisibility(View.VISIBLE);

        runnable = new Runnable() {
            @Override
            public void run() {
                staticImageCloseButton.setVisibility(View.VISIBLE);
            }
        };

        handler.postDelayed(runnable, 5000);

        staticImageView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                try {
                    interstitialAd.getAdListener().onAdClicked();
                    Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(adData.getClickUrl()));
                    activity.startActivity(browserIntent);
                } catch (NullPointerException | ActivityNotFoundException e) {
                    e.printStackTrace();
                }
            }
        });

        Picasso.get().load(Uri.parse(adData.getAdm())).into(staticImageView, new Callback() {
            @Override
            public void onSuccess() {
                interstitialAd.getAdListener().onAdDisplayed();

                if (burl != null) {
                    TrackingRequest trackingRequest = new TrackingRequest(activity, burl);
                    trackingRequest.loadRequest().subscribeOn(Schedulers.computation())
                            .observeOn(AndroidSchedulers.mainThread())
                            .subscribe(trackingRequest.requestObserver);
                }
            }

            @Override
            public void onError(Exception e) {
                interstitialAd.getAdListener().onAdFailedToDisplay(ErrorCodes.AssetDownload);
                activity.finish();
            }
        });

        staticImageCloseButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                interstitialAd.getAdListener().onAdClosed();
                activity.finish();
            }
        });
    }

    /**
     * Method used to render a WebView based on the information received from the ad data.
     * @param adData
     */
    private void renderHTML(AdData adData) {
        htmlLayout.setVisibility(View.VISIBLE);

        String content = AdWebViewPreparer.prepareWebViewUrl(adData.getAdm());
        adWebView.setWebViewClient(new AdWebViewClient(activity, false, interstitialAd, null));
        adWebView.setWebChromeClient(new AdWebChromeClient());
        adWebView.setHorizontalScrollBarEnabled(false);
        adWebView.setVerticalScrollBarEnabled(false);
        adWebView.loadDataWithBaseURL(AdWebViewPreparer.HTML_BASE_URL, content, "text/html", "UTF-8", null);

        runnable = new Runnable() {
            @Override
            public void run() {
                htmlCloseButton.setVisibility(View.VISIBLE);
            }
        };

        handler.postDelayed(runnable, 5000);

        htmlCloseButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                interstitialAd.getAdListener().onAdClosed();
                activity.finish();
            }
        });
    }

    /**
     * Method used to render a non-vast video based on the ad data.
     * @param adData
     */
    private void renderVideo(final AdData adData) {
        if (!shouldRenderVideo) {
            return;
        }

        videoLayout.setVisibility(View.VISIBLE);

        runnable = new Runnable() {
            @Override
            public void run() {
                videoCloseButton.setVisibility(View.VISIBLE);
            }
        };

        handler.postDelayed(runnable, 5000);

        BaseAdListener baseAdListener = null;

        switch (adType) {
            case StaticValues.INTERSTITIAL:
                if (interstitialAd != null) {
                    baseAdListener = interstitialAd.getAdListener();
                }
                break;
            case StaticValues.REWARDED_VIDEO:
                if (rewardedVideoAd != null) {
                    baseAdListener = rewardedVideoAd.getAdListener();
                }
                break;
        }

        if (baseAdListener == null) {
            return;
        }

        final BaseAdListener finalBaseAdListener = baseAdListener;

        videoPlayer.setUseController(false);
        HttpProxyCacheServer proxy = GamedockAds.getInstance().getProxy();
        videoPlayerCacheListener = new CacheListener() {
            @Override
            public void onCacheAvailable(File file, String url, int percentsAvailable) {

            }
        };
        proxy.registerCacheListener(videoPlayerCacheListener, adData.getAdm());
        String proxyUrl = proxy.getProxyUrl(adData.getAdm());

        simpleVideoPlayer = new SimpleExoPlayer.Builder(activity).build();

        simpleVideoPlayer.addAnalyticsListener(new AnalyticsListener() {
            @Override
            public void onPlayerError(EventTime eventTime, ExoPlaybackException error) {
                finalBaseAdListener.onAdFailedToDisplay(ErrorCodes.PlayerError);
            }

            @Override
            public void onLoadError(EventTime eventTime, LoadEventInfo loadEventInfo, MediaLoadData mediaLoadData, IOException error, boolean wasCanceled) {
                finalBaseAdListener.onAdFailedToDisplay(ErrorCodes.AssetDownload);
            }

            @Override
            public void onPlaybackStateChanged(EventTime eventTime, int state) {
                if (state == ExoPlayer.STATE_ENDED) {
                    if (shouldGiveReward) {
                        RewardedVideoAdListener rewardedVideoAdListener = null;
                        if (finalBaseAdListener instanceof RewardedVideoAdListener) {
                            rewardedVideoAdListener = (RewardedVideoAdListener) finalBaseAdListener;
                            rewardedVideoAdListener.onAdRewardEarned();
                        }
                    }

                    finalBaseAdListener.onAdClosed();
                    activity.finish();
                }
            }
        });

        videoPlayer.setPlayer(simpleVideoPlayer);

        DataSource.Factory dataSourceFactory = new DefaultDataSourceFactory(this, Util.getUserAgent(this, "GamedockAdsInfo"));
        ProgressiveMediaSource.Factory mediaSourceFactory = new ProgressiveMediaSource.Factory(dataSourceFactory);
        MediaSource videoSource = mediaSourceFactory.createMediaSource(Uri.parse(proxyUrl));

        videoPlayerClickButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                try {
                    Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(adData.getClickUrl()));
                    activity.startActivity(browserIntent);
                    finalBaseAdListener.onAdClicked();
                } catch (NullPointerException | ActivityNotFoundException e) {
                    e.printStackTrace();
                }
            }
        });

        videoCloseButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                finalBaseAdListener.onAdClosed();
                activity.finish();
            }
        });

        simpleVideoPlayer.prepare(videoSource);
        simpleVideoPlayer.setPlayWhenReady(true);
    }

    /**
     * Method used to render a VAST compliant video.
     */
    private void renderVastPlayer() {
        if (!shouldRenderVAST) {
            return;
        }

        BaseAdListener baseAdListener = null;

        switch (adType) {
            case StaticValues.INTERSTITIAL:
                if (interstitialAd != null) {
                    baseAdListener = interstitialAd.getAdListener();
                }
                break;
            case StaticValues.REWARDED_VIDEO:
                if (rewardedVideoAd != null) {
                    baseAdListener = rewardedVideoAd.getAdListener();
                }
                break;
        }

        if (baseAdListener == null) {
            return;
        }

        final BaseAdListener finalBaseAdListener = baseAdListener;

        if (vastContent == null) {
            finalBaseAdListener.onAdFailedToDisplay(ErrorCodes.EmptyAdData);
            activity.finish();
            return;
        }

        try {
            ImaAdsLoader.Builder builder = new ImaAdsLoader.Builder(activity);

            builder.setAdEventListener(new AdEvent.AdEventListener() {
                @Override
                public void onAdEvent(AdEvent adEvent) {
                    switch (adEvent.getType()) {
                        case STARTED:
                            finalBaseAdListener.onAdDisplayed();

                            if (burl != null) {
                                TrackingRequest trackingRequest = new TrackingRequest(activity, burl);
                                trackingRequest.loadRequest().subscribeOn(Schedulers.computation())
                                        .observeOn(AndroidSchedulers.mainThread())
                                        .subscribe(trackingRequest.requestObserver);
                            }
                            break;
                        case FIRST_QUARTILE:

                            break;
                        case MIDPOINT:

                            break;
                        case THIRD_QUARTILE:
                            shouldGiveReward = true;
                            break;
                        case ALL_ADS_COMPLETED:
                            vastPlayer.setVisibility(View.INVISIBLE);
                            if (shouldGiveReward) {
                                RewardedVideoAdListener rewardedVideoAdListener = null;
                                if (finalBaseAdListener instanceof RewardedVideoAdListener) {
                                    rewardedVideoAdListener = (RewardedVideoAdListener) finalBaseAdListener;
                                    rewardedVideoAdListener.onAdRewardEarned();
                                }
                            }

                            finalBaseAdListener.onAdClosed();
                            activity.finish();
                            break;
                        case CLICKED:
                            finalBaseAdListener.onAdClicked();
                            break;
                    }
                }
            });
            adsLoader = builder.buildForAdsResponse(vastContent);

            vastLayout.setVisibility(View.VISIBLE);
            vastPlayer.setUseController(false);

            simpleVastPlayer = new SimpleExoPlayer.Builder(this).build();
            vastPlayer.setPlayer(simpleVastPlayer);
            adsLoader.setPlayer(simpleVastPlayer);

            DataSource.Factory dataSourceFactory = new DefaultDataSourceFactory(this, Util.getUserAgent(this, "GamedockAdsInfo"));
            ProgressiveMediaSource.Factory mediaSourceFactory = new ProgressiveMediaSource.Factory(dataSourceFactory);
            MediaSource mediaSource = mediaSourceFactory.createMediaSource(Uri.parse("asset:///placeholder.mp4"));

            AdsMediaSource adsMediaSource = new AdsMediaSource(mediaSource, dataSourceFactory, adsLoader, vastPlayer);
            adsMediaSource.addEventListener(handler, new MediaSourceEventListener() {
                @Override
                public void onLoadError(int windowIndex, @Nullable MediaSource.MediaPeriodId mediaPeriodId, LoadEventInfo loadEventInfo, MediaLoadData mediaLoadData, IOException error, boolean wasCanceled) {
                    finalBaseAdListener.onAdFailedToDisplay(ErrorCodes.PlayerError);
                    activity.finish();
                }
            });
            simpleVastPlayer.prepare(adsMediaSource);
            simpleVastPlayer.setPlayWhenReady(true);
        } catch (Exception e) {
            e.printStackTrace();
            finalBaseAdListener.onAdFailedToDisplay(ErrorCodes.EmptyAdData);
            activity.finish();
        }
    }

    /**
     * Method used to check if the parameters sent to the activity are correct.
     * @param extras
     * @return
     */
    private boolean verifyActivityParameters(Bundle extras) {
        if (extras != null) {
            adType = extras.getString("adType", null);
            provider = AdProvider.valueOf(extras.getString("provider", "GAMEDOCK"));
        } else {
            switch (adType) {
                case StaticValues.INTERSTITIAL:
                    interstitialAd.getAdListener().onAdFailedToDisplay(ErrorCodes.EmptyAdData);
                    break;
                case StaticValues.REWARDED_VIDEO:
                    rewardedVideoAd.getAdListener().onAdFailedToDisplay(ErrorCodes.EmptyAdData);
                    break;
            }

            return false;
        }

        if (adType == null) {
            if (interstitialAd != null) {
                interstitialAd.getAdListener().onAdFailedToDisplay(ErrorCodes.EmptyAdData);
            } else if (rewardedVideoAd != null) {
                rewardedVideoAd.getAdListener().onAdFailedToDisplay(ErrorCodes.EmptyAdData);
            }
            return false;
        }

        return true;
    }

    /**
     * Method used to configure the privacy policy button in the activity view.
     */
    private void configurePrivacyPolicy() {
        String text;
        String url;
        Drawable logo;

        switch (provider) {
            case IMPROVE_DIGITAL:
                text = "Ads by Improve Digital";
                url = ImproveDigitalProvider.PRIVACY_POLICY_URL;
                logo = activity.getResources().getDrawable(R.drawable.ic_improve_digital);
                break;
            case GAMEDOCK:
            default:
                text = "Ads by Gamedock";
                url = "https://azerion.com/business/mobile-apps-privacy-policy.html";
                logo = activity.getResources().getDrawable(R.drawable.ic_close);
                break;
        }

        privacyPolicyImage.setBackground(logo);
        privacyPolicyText.setText(text);
        final String finalUrl = url;
        privacyPolicy.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                try {
                    Intent browserIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(finalUrl));
                    activity.startActivity(browserIntent);
                } catch (NullPointerException | ActivityNotFoundException e) {
                    e.printStackTrace();
                }
            }
        });
    }

    @Override
    protected void onStart() {
        super.onStart();

        if (Util.SDK_INT > 23) {
            renderVastPlayer();
            if (vastPlayer != null) {
                vastPlayer.onResume();
            }
        }
    }

    @Override
    protected void onResume() {
        super.onResume();

        View decorView = getWindow().getDecorView();
        // Hide the status bar.
        int uiOptions = View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                | View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION
                | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION
                | View.SYSTEM_UI_FLAG_FULLSCREEN
                | View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY;
        decorView.setSystemUiVisibility(uiOptions);

        try {
            int currentOrientation = this.getResources().getConfiguration().orientation;
            if (currentOrientation == Configuration.ORIENTATION_PORTRAIT) {
                this.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT);
            } else {
                this.setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }


        if (Util.SDK_INT <= 23 || simpleVastPlayer == null) {
            renderVastPlayer();
            if (vastPlayer != null) {
                vastPlayer.onResume();
            }
        }
    }

    @Override
    protected void onPause() {
        super.onPause();

        if (Util.SDK_INT <= 23) {
            if (vastPlayer != null) {
                vastPlayer.onPause();
            }
            releaseVastPlayer();
        }
    }

    @Override
    protected void onStop() {
        super.onStop();

        if (Util.SDK_INT > 23) {
            if (vastPlayer != null) {
                vastPlayer.onPause();
            }
            releaseVastPlayer();
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();

        interstitialAd = null;
        rewardedVideoAd = null;

        adType = null;
        vastContent = null;
        shouldRenderVAST = false;

        if (handler != null) {
            handler.removeCallbacks(runnable);
        }

        if (adsLoader != null) {
            adsLoader.release();
        }

        if (simpleVastPlayer != null) {
            simpleVastPlayer.release();
        }

        if (simpleVideoPlayer != null) {
            simpleVideoPlayer.release();
        }

        if (videoPlayerCacheListener != null) {
            GamedockAds.getInstance().getProxy().unregisterCacheListener(videoPlayerCacheListener);
        }
    }

    private void releaseVastPlayer() {
        if (!shouldRenderVAST) {
            return;
        }
        adsLoader.setPlayer(null);
        vastPlayer.setPlayer(null);
        simpleVastPlayer.release();
        simpleVastPlayer = null;
    }

    public static void setInterstitialAd(InterstitialAd interstitialAd) {
        GamedockAdsActivity.interstitialAd = interstitialAd;
    }

    public static void setRewardedVideoAd(RewardedVideoAd rewardedVideoAd) {
        GamedockAdsActivity.rewardedVideoAd = rewardedVideoAd;
    }
}