package io.gamedock.sdk.ads.core.rewarded;

import android.content.Context;
import android.content.Intent;

import com.google.gson.Gson;

import io.gamedock.sdk.ads.GamedockAds;
import io.gamedock.sdk.ads.core.base.AdData;
import io.gamedock.sdk.ads.core.base.BaseAd;
import io.gamedock.sdk.ads.core.request.AdType;
import io.gamedock.sdk.ads.core.request.RequestListener;
import io.gamedock.sdk.ads.core.request.RequestType;
import io.gamedock.sdk.ads.core.ui.GamedockAdsActivity;
import io.gamedock.sdk.ads.providers.AdProvider;
import io.gamedock.sdk.ads.providers.improvedigital.ImproveDigitalRequest;
import io.gamedock.sdk.ads.providers.improvedigital.ImproveDigitalResponse;
import io.gamedock.sdk.ads.utils.error.ErrorCodes;
import io.gamedock.sdk.ads.utils.logging.LoggingUtilAds;
import io.gamedock.sdk.ads.utils.values.StaticValues;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.schedulers.Schedulers;

/**
 * Logic class for handling all rewarded video operations for the module.
 * All relevant callbacks are fired through the RewardedVideoAdListener.
 * The RewardedVideoAd is bound by a placement id and an ad provider (ex.:Improve Digital).
 */
public class RewardedVideoAd extends BaseAd {
    private RewardedVideoAdListener adListener;

    public RewardedVideoAd(AdProvider network, String placementId, RewardedVideoAdListener adListener) {
        super(network, new String[]{placementId});

        if (adListener == null) {
            throw new IllegalArgumentException("Ad listener cannot be null");
        }
        this.adListener = adListener;
    }

    /**
     * Method used to load a rewarded video ad based on the supplied placement id.
     */
    @Override
    public void load() {
        if (isLoading) {
            return;
        }

        if (!GamedockAds.getInstance().isInitialized()) {
            adListener.onAdFailedToLoad(ErrorCodes.NotInitialized);
            return;
        }

        Context context = GamedockAds.getInstance().getContext();

        if (context == null) {
            adListener.onAdFailedToLoad(ErrorCodes.NullContext);
            return;
        }

        if (placementIds == null || placementIds[0] == null || placementIds[0].isEmpty()) {
            adListener.onAdFailedToLoad(ErrorCodes.NullPlacementId);
            return;
        }

        if (provider == AdProvider.IMPROVE_DIGITAL) {
            if (!GamedockAds.getInstance().isProviderInitialized(AdProvider.IMPROVE_DIGITAL)) {
                adListener.onAdFailedToLoad(ErrorCodes.NotInitialized);
                return;
            }

            ImproveDigitalRequest improveDigitalRequest = new ImproveDigitalRequest(context, placementIds[0], AdType.REWARDED_VIDEO, RequestType.VAST, new RequestListener() {
                @Override
                public void onSuccess(String responseJSON) {
                    ImproveDigitalResponse improveDigitalResponse = new ImproveDigitalResponse(responseJSON, AdType.REWARDED_VIDEO);

                    if (improveDigitalResponse.isError()) {
                        adListener.onAdFailedToLoad(improveDigitalResponse.getError());
                    } else {
                        adData = new AdData();

                        adData.populateWithImproveDigitalResponse(improveDigitalResponse);

                        isLoading = false;
                        isLoaded = true;

                        adListener.onAdLoaded();
                    }
                }

                @Override
                public void onFailure(ErrorCodes error) {
                    isLoading = false;
                    adListener.onAdFailedToLoad(error);
                }
            }, null);

            improveDigitalRequest.loadRequest().subscribeOn(Schedulers.computation())
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(improveDigitalRequest.requestObserver);
            isLoading = true;
        } else {
            adListener.onAdFailedToLoad(ErrorCodes.UnknownProvider);
        }
    }

    /**
     * Method used to show the loaded rewarded video ad. If an ad has not been loaded an error callback will be fired.
     */
    @Override
    public void show() {
        if (!GamedockAds.getInstance().isInitialized()) {
            adListener.onAdFailedToLoad(ErrorCodes.NotInitialized);
            return;
        }

        if (!isLoaded) {
            adListener.onAdFailedToDisplay(ErrorCodes.NotLoaded);
            return;
        }

        Context context = GamedockAds.getInstance().getContext();

        if (context == null) {
            adListener.onAdFailedToDisplay(ErrorCodes.NullContext);
            return;
        }

        LoggingUtilAds.d("Showing rewarded ad with data: " + new Gson().toJson(adData));

        GamedockAdsActivity.setRewardedVideoAd(this);

        Intent intent = new Intent(context, GamedockAdsActivity.class);
        intent.putExtra("adType", StaticValues.REWARDED_VIDEO);
        intent.putExtra("provider", getProvider().toString());
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

        context.startActivity(intent);
    }

    public RewardedVideoAdListener getAdListener() {
        return adListener;
    }

    public void setAdListener(RewardedVideoAdListener adListener) {
        this.adListener = adListener;
    }
}
