package io.gamedock.sdk.ads.admob.fyber;


import android.content.Context;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;
import android.view.ViewGroup;
import android.widget.RelativeLayout;

import com.fyber.inneractive.sdk.external.InneractiveAdManager;
import com.fyber.inneractive.sdk.external.InneractiveAdRequest;
import com.fyber.inneractive.sdk.external.InneractiveAdSpot;
import com.fyber.inneractive.sdk.external.InneractiveAdSpotManager;
import com.fyber.inneractive.sdk.external.InneractiveAdViewEventsListener;
import com.fyber.inneractive.sdk.external.InneractiveAdViewUnitController;
import com.fyber.inneractive.sdk.external.InneractiveErrorCode;
import com.fyber.inneractive.sdk.external.InneractiveMediationDefs;
import com.fyber.inneractive.sdk.external.InneractiveMediationName;
import com.fyber.inneractive.sdk.external.InneractiveUnitController.AdDisplayError;
import com.fyber.inneractive.sdk.external.InneractiveUserConfig;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.mediation.customevent.CustomEventBanner;
import com.google.android.gms.ads.mediation.customevent.CustomEventBannerListener;

import org.json.JSONObject;

import java.util.Set;

/**
 * Inneractive Admob/DFP banner plugin class
 */
public class InneractiveBannerForGoogle implements CustomEventBanner {

  private static final String TAG = InneractiveBannerForGoogle.class.getSimpleName();

  // Set your Inneractive's spot ID!
  private static String SAMPLE_BANNER_SPOT_ID = "Set_Your_Inneractive_Spot_Id"; //for example, inneractive's spot id for testing: "150942"

  // Definitions
  private final static InneractiveMediationName MEDIATOR_NAME = InneractiveMediationName.ADMOB;

  // Members
  /**
   * The Spot object for the banner
   */
  InneractiveAdSpot mBannerSpot;
  /**
   * The event listener of the Ad
   */
  InneractiveAdViewEventsListener mAdListener;
  /**
   * Flag to indicate external app been opened
   */
  Boolean mOpenedExternalApp = false;
  private String mAppId;
  /**
   * Called by the Admob/DFP infra-structure when Admob/DFP requests a banner from Inneractive
   *
   * @param context Android context
   * @param customEventListener Callback interface
   * @param serverParameter As defined in Admob/DFP console
   * @param size Requested ad size
   * @param mediationAdRequest Additional ad data
   * @param customEventExtras Extra data can be provided
   */
  @Override
  public void requestBannerAd(
      final Context context,
      final CustomEventBannerListener customEventListener,
      String serverParameter,
      com.google.android.gms.ads.AdSize size,
      com.google.android.gms.ads.mediation.MediationAdRequest mediationAdRequest,
      Bundle customEventExtras) {

    // You must set the mediation name to either AdMob or DFP
    // MEDIATOR_NAME = InneractiveMediationName.ADMOB;
    // Or
    // MEDIATOR_NAME = InneractiveMediationName.DFP;
    if (MEDIATOR_NAME == null || (MEDIATOR_NAME != InneractiveMediationName.ADMOB
        && MEDIATOR_NAME != InneractiveMediationName.DFP)) {
      Log.i(TAG,
          InneractiveMediationDefs.IA_LOG_FOR_ADMOB_BANNER + " - you must set mediation name");
      customEventListener.onAdFailedToLoad(AdRequest.ERROR_CODE_INVALID_REQUEST);
      return;
    }

    String resultAppId = null;
    String resultSpotId = null;
    String resultKeywords = null;
    //DFP note - parameter string in the DFP console needs to be escaped, or else it will break
    //the google dfp parser. escaped example - {\"key\":\"value\"}
    //Escaping the JSON is not needed for admob server parameter.
    JSONObject serverExtras = null;

    if (serverParameter != null) {
      try {
        serverExtras = new JSONObject(serverParameter);
      } catch (Exception ignored) {
      }
    }

    if (serverExtras != null) {
      resultSpotId = serverExtras.optString(InneractiveMediationDefs.REMOTE_KEY_SPOT_ID);
      resultAppId = serverExtras.optString(InneractiveMediationDefs.REMOTE_KEY_APP_ID);
      resultKeywords = serverExtras.optString(InneractiveMediationDefs.REMOTE_KEY_KEYWORDS);
    }

    mAppId = resultAppId;

    if (!InneractiveAdManager.wasInitialized()) {
      InneractiveAdManager.initialize(context, mAppId);
    }

    //Make sure we have a spotId from the request.
    if (TextUtils.isEmpty(resultSpotId)) {
      resultSpotId = SAMPLE_BANNER_SPOT_ID;
      if (TextUtils.isEmpty(resultSpotId)) {
        customEventListener.onAdFailedToLoad(AdRequest.ERROR_CODE_INVALID_REQUEST);
        return;
      }
    }

    StringBuilder iaKeyWords = new StringBuilder();
    String iaZip = null;

    // Check whether the Ad Request contains user data which can be added to Inneractive Request
    if (mediationAdRequest != null) {
      Set<String> keyWords = mediationAdRequest.getKeywords();
      if (keyWords != null) {
        for (String key : keyWords) {
          if (iaKeyWords.length() > 0) {
            iaKeyWords.append(",");
          }

          iaKeyWords.append(key);
        }
      }
    }

    if (customEventExtras != null) {
      iaZip = customEventExtras.getString(InneractiveMediationDefs.KEY_ZIPCODE);
    }

    // Destroy previous Spot if exists
    if (mBannerSpot != null) {
      mBannerSpot.destroy();
    }

    mBannerSpot = InneractiveAdSpotManager.get().createSpot();
    // Set your mediation name
    mBannerSpot.setMediationName(MEDIATOR_NAME);

    InneractiveAdViewUnitController controller = new InneractiveAdViewUnitController();
    mBannerSpot.addUnitController(controller);

    InneractiveAdRequest request = new InneractiveAdRequest(resultSpotId);
    request.setUserParams(new InneractiveUserConfig()
        .setZipCode(iaZip));

    if (!TextUtils.isEmpty(resultKeywords)) {
      if (iaKeyWords.length() > 0) {
        iaKeyWords.append(",");
      }
      iaKeyWords.append(resultKeywords);
    }

    if (iaKeyWords.length() > 0) {
      request.setKeywords(iaKeyWords.toString());
    }

    mBannerSpot.setRequestListener(new InneractiveAdSpot.RequestListener() {
      @Override
      public void onInneractiveSuccessfulAdRequest(InneractiveAdSpot adSpot) {
        if (adSpot != mBannerSpot) {
          Log.d(TAG, "Wrong Banner Spot: Received - " + adSpot + ", Actual - " + mBannerSpot);
          return;
        }

        // Create a parent layout for the Banner Ad
        ViewGroup layout = new RelativeLayout(context);
        InneractiveAdViewUnitController controller = (InneractiveAdViewUnitController) mBannerSpot
            .getSelectedUnitController();

          mAdListener = new InneractiveAdViewEventsListener() {
              @Override
              public void onAdImpression(InneractiveAdSpot adSpot) {
                  Log.d(TAG, InneractiveMediationDefs.IA_LOG_FOR_ADMOB_BANNER + " - onAdImpression");
              }

              @Override
              public void onAdClicked(InneractiveAdSpot adSpot) {
                  customEventListener.onAdClicked();
                  customEventListener.onAdOpened();
              }

              @Override
              public void onAdWillCloseInternalBrowser(InneractiveAdSpot adSpot) {
                  Log.d(TAG, InneractiveMediationDefs.IA_LOG_FOR_ADMOB_BANNER
                                  + " - inneractiveInternalBrowserDismissed");
                  customEventListener.onAdClosed();
                  mOpenedExternalApp = false;
              }

              @Override
              public void onAdWillOpenExternalApp(InneractiveAdSpot adSpot) {
                  Log.d(TAG, InneractiveMediationDefs.IA_LOG_FOR_ADMOB_BANNER
                                  + " - inneractiveAdWillOpenExternalApp");
                  customEventListener.onAdLeftApplication();
                  mOpenedExternalApp = true;
              }

              @Override
              public void onAdEnteredErrorState(InneractiveAdSpot adSpot, AdDisplayError error) {
                Log.d(TAG, InneractiveMediationDefs.IA_LOG_FOR_ADMOB_BANNER
                        + " - onAdEnteredErrorState - " + error.getMessage());
              }

            @Override
              public void onAdExpanded(InneractiveAdSpot adSpot) {
                  Log.d(TAG, InneractiveMediationDefs.IA_LOG_FOR_ADMOB_BANNER + " - onAdExpanded");
              }

              @Override
              public void onAdResized(InneractiveAdSpot adSpot) {
                  Log.d(TAG, InneractiveMediationDefs.IA_LOG_FOR_ADMOB_BANNER + " - onAdResized");
              }

              @Override
              public void onAdCollapsed(InneractiveAdSpot adSpot) {
                  Log.d(TAG, InneractiveMediationDefs.IA_LOG_FOR_ADMOB_BANNER + " - onAdCollapsed");
              }
          };

        controller.setEventsListener(mAdListener);

        controller.bindView(layout);
        customEventListener.onAdLoaded(layout);
      }

      @Override
      public void onInneractiveFailedAdRequest(InneractiveAdSpot adSpot,
          InneractiveErrorCode errorCode) {
        Log.d(TAG,
            "Failed loading Banner! with error: " + errorCode);
        if (errorCode == InneractiveErrorCode.CONNECTION_ERROR
            || errorCode == InneractiveErrorCode.CONNECTION_TIMEOUT) {
          customEventListener.onAdFailedToLoad(AdRequest.ERROR_CODE_NETWORK_ERROR);
        } else if (errorCode == InneractiveErrorCode.NO_FILL) {
          customEventListener.onAdFailedToLoad(AdRequest.ERROR_CODE_NO_FILL);
        } else {
          customEventListener.onAdFailedToLoad(AdRequest.ERROR_CODE_INTERNAL_ERROR);
        }
      }
    });

    mBannerSpot.requestAd(request);
  }

  /**
   * Called by Admob/DFP when an ad should be destroyed
   * <br>Destroy the underline Inneractive ad
   */
  @Override
  public void onDestroy() {
    Log.d(TAG, "Banner - destroy");
    if (mBannerSpot != null) {
      mBannerSpot.destroy();
      mBannerSpot = null;
    }

    if (mAdListener != null) {
        mAdListener = null;
    }

    mOpenedExternalApp = false;
  }

  /**
   * Called by Admob/DFP when an ad should be paused. Usually due to the Activity:onPause event
   * <br>Can be used for example to stop a video ad
   */
  @Override
  public void onPause() {
    // Currently not implemented
    Log.d(TAG, "Banner - onPause");
  }

  /**
   * Called by Admob/DFP when an ad should be resumed. Usually due to the Activity:onResume event
   * <br>Can be used for example to resume a video ad
   */
  @Override
  public void onResume() {
    // Currently not implemented
    Log.d(TAG, "Banner - onResume");
    if (mOpenedExternalApp && mAdListener != null) {
       mAdListener.onAdWillCloseInternalBrowser(null);
    }
  }
}
