package io.gamedock.sdk.ads.admob.fyber;

import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;

import com.fyber.inneractive.sdk.external.InneractiveAdManager;
import com.fyber.inneractive.sdk.external.InneractiveAdRequest;
import com.fyber.inneractive.sdk.external.InneractiveAdSpot;
import com.fyber.inneractive.sdk.external.InneractiveAdSpotManager;
import com.fyber.inneractive.sdk.external.InneractiveErrorCode;
import com.fyber.inneractive.sdk.external.InneractiveFullscreenAdEventsListener;
import com.fyber.inneractive.sdk.external.InneractiveFullscreenUnitController;
import com.fyber.inneractive.sdk.external.InneractiveFullscreenVideoContentController;
import com.fyber.inneractive.sdk.external.InneractiveMediationDefs;
import com.fyber.inneractive.sdk.external.InneractiveMediationName;
import com.fyber.inneractive.sdk.external.InneractiveUnitController.AdDisplayError;
import com.fyber.inneractive.sdk.external.InneractiveUserConfig;
import com.fyber.inneractive.sdk.external.VideoContentListener;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.mediation.MediationAdRequest;
import com.google.android.gms.ads.reward.RewardItem;
import com.google.android.gms.ads.reward.mediation.MediationRewardedVideoAdAdapter;
import com.google.android.gms.ads.reward.mediation.MediationRewardedVideoAdListener;

import org.json.JSONObject;

import java.util.Set;

/**
 * Rewarded adapter (AdMob)
 * Please note - this example assumes you have the
 * 'Override reward settings from third-party ad networks when using AdMob mediation'
 * Ticked in the ad-mob unit settings
 * TODO: OnContextChangedListener as needed
 */

public class InneractiveRewardedVideoForGoogle implements MediationRewardedVideoAdAdapter {

  private static final String TAG = InneractiveRewardedVideoForGoogle.class.getSimpleName();

  // Set your Inneractive's spot ID!
  private static String SAMPLE_REWARDED_SPOT_ID = "Set_Your_Inneractive_Spot_Id"; //for example, inneractive's spot id for testing: "150949"

  private static InneractiveMediationName MEDIATOR_NAME = InneractiveMediationName.ADMOB;

  InneractiveAdSpot mRewardedSpot;
  Activity mParentActivity;
  private MediationRewardedVideoAdListener mRewardedListener;
  StringBuilder iaKeyWords = new StringBuilder();
  private String mSpotId;
  private String mAppId;
  private boolean mIsReceivedRewardItem = false;


  /** In this method, the inneractive sdk is initialized */
  @Override
  public void initialize(Context context,
      MediationAdRequest mediationAdRequest,
      String unused,
      MediationRewardedVideoAdListener listener,
      Bundle serverParameters,
      Bundle mediationExtras) {

    // You must set the mediation name to either AdMob or DFP
    // MEDIATOR_NAME = InneractiveMediationName.ADMOB;
    // Or
    // MEDIATOR_NAME = InneractiveMediationName.DFP;
    if (MEDIATOR_NAME == null || (MEDIATOR_NAME != InneractiveMediationName.ADMOB && MEDIATOR_NAME != InneractiveMediationName.DFP)) {
      Log.i(TAG , InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL + " - you must set mediation name");
      listener.onInitializationFailed(this, AdRequest.ERROR_CODE_INVALID_REQUEST);
      return;
    }

    if (!(context instanceof Activity)) {
      // Context not an Activity context, log the reason for failure and fail the
      // initialization.
      Log.i(TAG , "Inneractive SDK requires an Activity context to initialize");
      listener.onInitializationFailed(this, AdRequest.ERROR_CODE_INVALID_REQUEST);
      return;
    }

    String parameter = serverParameters.getString(MediationRewardedVideoAdAdapter.CUSTOM_EVENT_SERVER_PARAMETER_FIELD);
    mParentActivity = (Activity) context;

    String resultSpotId = null;
    String resultAppId = null;
    String resultKeywords = null;
    JSONObject serverExtras = null;
    if (parameter != null) {
      try {
        serverExtras = new JSONObject(parameter);
      } catch (Exception ignored) {}
    }

    if (serverExtras != null) {
      resultSpotId = serverExtras.optString(InneractiveMediationDefs.REMOTE_KEY_SPOT_ID);
      resultAppId = serverExtras.optString(InneractiveMediationDefs.REMOTE_KEY_APP_ID);
      resultKeywords = serverExtras.optString(InneractiveMediationDefs.REMOTE_KEY_KEYWORDS);
    }

    //Make sure we have a spotId from the request.
    if (TextUtils.isEmpty(resultSpotId)){
      resultSpotId = SAMPLE_REWARDED_SPOT_ID;
      if (TextUtils.isEmpty(resultSpotId)) {
        listener.onInitializationFailed(this, AdRequest.ERROR_CODE_INVALID_REQUEST);
        return;
      }
    }

    if (!TextUtils.isEmpty(resultKeywords)) {
      if (iaKeyWords.length() > 0) iaKeyWords.append(",");
      iaKeyWords.append(resultKeywords);
    }

    mRewardedListener = listener;
    mSpotId = resultSpotId;
    mAppId = resultAppId;

    if (!InneractiveAdManager.wasInitialized()) {
      InneractiveAdManager.initialize(context, mAppId);
    }

    listener.onInitializationSucceeded(this);

  }

  @Override
  public void loadAd(MediationAdRequest mediationAdRequest,
      Bundle serverParameters,
      Bundle mediationExtras) {

    String iaZip = null;

    // Check whether the Ad Request contains user data which can be added to Inneractive Request
    if (mediationAdRequest != null) {
      Set<String> keyWords = mediationAdRequest.getKeywords();
      if (keyWords != null) {
        for (String key : keyWords) {
          if (iaKeyWords.length() > 0) {
            iaKeyWords.append(",");
          }

          iaKeyWords.append(key);
        }
      }
    }

    if (mediationExtras != null) {
      iaZip = mediationExtras.getString(InneractiveMediationDefs.KEY_ZIPCODE);
    }

    if (mRewardedSpot != null) {
      mRewardedSpot.destroy();
    }

    mRewardedSpot = InneractiveAdSpotManager.get().createSpot();
    // Set the mediation name
    mRewardedSpot.setMediationName(MEDIATOR_NAME);

    InneractiveFullscreenUnitController fullscreenUnitController = new InneractiveFullscreenUnitController();
    mRewardedSpot.addUnitController(fullscreenUnitController);

    InneractiveAdRequest request = new InneractiveAdRequest(mSpotId);
    request.setUserParams( new InneractiveUserConfig()
        .setZipCode(iaZip));
    if (iaKeyWords != null && iaKeyWords.length() > 0) {
      request.setKeywords(iaKeyWords.toString());
    }

    // Load ad
    mRewardedSpot.setRequestListener(new InneractiveAdSpot.RequestListener() {

      /**
       * Called by Inneractive when an interstitial is ready for display
       * @param adSpot Spot object
       */
      @Override
      public void onInneractiveSuccessfulAdRequest(InneractiveAdSpot adSpot) {
        Log.d(TAG, InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL + " - inneractiveInterstitialLoaded");
        mRewardedListener.onAdLoaded(InneractiveRewardedVideoForGoogle.this);
      }

      /**
       * Called by Inneractive an interstitial fails loading
       * @param adSpot Spot object
       * @param errorCode the failure's error.
       */
      @Override
      public void onInneractiveFailedAdRequest(InneractiveAdSpot adSpot, InneractiveErrorCode errorCode) {
        Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "Failed loading interstitial! with error: " + errorCode);
        if (errorCode == InneractiveErrorCode.CONNECTION_ERROR) {
          mRewardedListener.onAdFailedToLoad(InneractiveRewardedVideoForGoogle.this, AdRequest.ERROR_CODE_NETWORK_ERROR);
        } else if  (errorCode == InneractiveErrorCode.CONNECTION_TIMEOUT) {
          mRewardedListener.onAdFailedToLoad(InneractiveRewardedVideoForGoogle.this, AdRequest.ERROR_CODE_NETWORK_ERROR);
        } else if (errorCode == InneractiveErrorCode.NO_FILL) {
          mRewardedListener.onAdFailedToLoad(InneractiveRewardedVideoForGoogle.this, AdRequest.ERROR_CODE_NO_FILL);
        } else {
          mRewardedListener.onAdFailedToLoad(InneractiveRewardedVideoForGoogle.this, AdRequest.ERROR_CODE_INTERNAL_ERROR);
        }
      }
    });

    mRewardedSpot.requestAd(request);



  }

  @Override
  public void showVideo() {
    Log.d(TAG , InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL + " - showInterstitial");

    if (mRewardedSpot != null && mRewardedSpot.isReady()) {
      InneractiveFullscreenUnitController fullscreenUnitController = (InneractiveFullscreenUnitController)mRewardedSpot.getSelectedUnitController();
      fullscreenUnitController.setEventsListener(new InneractiveFullscreenAdEventsListener() {

        /**
         * Called by Inneractive when an interstitial ad activity is closed
         * @param adSpot Spot object
         */
        @Override
        public void onAdDismissed(InneractiveAdSpot adSpot) {
          mRewardedListener.onAdClosed(InneractiveRewardedVideoForGoogle.this);
          Log.i(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "onAdDismissed");
        }

        /**
         * Called by Inneractive when an interstitial ad activity is shown
         * @param adSpot Spot object
         */
        @Override
        public void onAdImpression(InneractiveAdSpot adSpot) {
          mRewardedListener.onAdOpened(InneractiveRewardedVideoForGoogle.this);
          mRewardedListener.onVideoStarted(InneractiveRewardedVideoForGoogle.this);
          Log.i(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "onAdImpression");
        }

        /**
         * Called by Inneractive when an interstitial ad is clicked
         * @param adSpot Spot object
         */
        @Override
        public void onAdClicked(InneractiveAdSpot adSpot) {
          mRewardedListener.onAdClicked(InneractiveRewardedVideoForGoogle.this);
          Log.i(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "onAdClicked");
        }

        /**
         * Called by Inneractive when an interstitial ad opened an external application
         * @param adSpot Spot object
         */
        @Override
        public void onAdWillOpenExternalApp(InneractiveAdSpot adSpot) {
          Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "inneractiveAdWillOpenExternalApp");
          mRewardedListener.onAdLeftApplication(InneractiveRewardedVideoForGoogle.this);
        }

        /**
         * Called when an ad has entered an error state, this will only happen when the ad is being shown
         *
         * @param adSpot the relevant ad spot
         */
        @Override
        public void onAdEnteredErrorState(InneractiveAdSpot adSpot, AdDisplayError error) {
          Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "onAdEnteredErrorState - " + error.getMessage());
        }

        /**
         * Called by Inneractive when Inneractive's internal browser, which was opened by this interstitial, was closed
         * @param adSpot Spot object
         */
        @Override
        public void onAdWillCloseInternalBrowser(InneractiveAdSpot adSpot) {
          Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "inneractiveInternalBrowserDismissed");
        }
      });

      // Add video content controller, for controlling video ads
      InneractiveFullscreenVideoContentController videoContentController = new InneractiveFullscreenVideoContentController();
      videoContentController.setEventsListener(new VideoContentListener() {

        @Override
        public void onProgress(int totalDurationInMsec, int positionInMsec) {
          Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "Interstitial: Got video content progress: total time = " + totalDurationInMsec + " position = " + positionInMsec);
        }

        /**
         * Called by inneractive when an Intersititial video ad was played to the end
         * <br>Can be used for incentive flow
         * <br>Note: This event does not indicate that the interstitial was closed
         */
        @Override
        public void onCompleted() {
          if(!mIsReceivedRewardItem) {
            mIsReceivedRewardItem = true;
            mRewardedListener.onRewarded(InneractiveRewardedVideoForGoogle.this, new RewardItem() {
              @Override
              public String getType() {
                return "";
              }

              @Override
              public int getAmount() {
                return 1;
              }
            });
          }
          Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "Interstitial: Got video content completed event");
        }

        @Override
        public void onPlayerError() {
          Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "Interstitial: Got video content play error event");
          mRewardedListener.onAdFailedToLoad(InneractiveRewardedVideoForGoogle.this, AdRequest.ERROR_CODE_INTERNAL_ERROR);
        }
      });

      // Now add the content controller to the unit controller
      fullscreenUnitController.addContentController(videoContentController);
      fullscreenUnitController.show((Activity)mParentActivity);
    } else {
      Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "The Interstitial ad is not ready yet.");
    }


  }

  @Override
  public boolean isInitialized() {
    return mSpotId != null;
  }

  @Override
  public void onDestroy() {
    if (mRewardedSpot != null) {
      mRewardedSpot.destroy();
      iaKeyWords = new StringBuilder();
      mRewardedSpot = null;
      mIsReceivedRewardItem = false;
    }
  }

  @Override
  public void onPause() {

  }

  @Override
  public void onResume() {

  }

}
