package io.gamedock.sdk.ads.admob.fyber;

import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;

import com.fyber.inneractive.sdk.external.InneractiveAdRequest;
import com.fyber.inneractive.sdk.external.InneractiveAdSpot;
import com.fyber.inneractive.sdk.external.InneractiveAdSpotManager;
import com.fyber.inneractive.sdk.external.InneractiveErrorCode;
import com.fyber.inneractive.sdk.external.InneractiveFullscreenAdEventsListener;
import com.fyber.inneractive.sdk.external.InneractiveFullscreenUnitController;
import com.fyber.inneractive.sdk.external.InneractiveFullscreenVideoContentController;
import com.fyber.inneractive.sdk.external.InneractiveMediationDefs;
import com.fyber.inneractive.sdk.external.InneractiveMediationName;
import com.fyber.inneractive.sdk.external.InneractiveUnitController.AdDisplayError;
import com.fyber.inneractive.sdk.external.InneractiveUserConfig;
import com.fyber.inneractive.sdk.external.VideoContentListener;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.mediation.customevent.CustomEventInterstitial;
import com.google.android.gms.ads.mediation.customevent.CustomEventInterstitialListener;

import org.json.JSONObject;

import java.util.Set;

/**
 * Inneractive Admob/DFP interstitial plugin class
 */
public class InneractiveInterstitialForGoogle implements CustomEventInterstitial {

  private static final String TAG = InneractiveInterstitialForGoogle.class.getSimpleName();

  // Set your Inneractive's spot ID!
  private static final String SAMPLE_INTERSTITIAL_SPOT_ID = "Set_Your_Inneractive_Spot_Id"; //for example, inneractive's spot id for testing: "150946"

  // Definitions
  private final static InneractiveMediationName MEDIATOR_NAME = InneractiveMediationName.ADMOB;


  // Members
  /**
   * Inneractived interstitial ad
   */
  InneractiveAdSpot mInterstitialSpot;
  /**
   * Context for showing the Ad
   */
  Context mContext;
  /**
   * The plugin fires events into Admob/DFP's callback listener
   */
  private CustomEventInterstitialListener mInterstitialListener;

  /**
   * Called by the Admob/DFP infra-structure when Admob/DFP requests an interstitial from
   * Inneractive
   *
   * @param context Android context
   * @param customEventListener Callback interface
   * @param serverParameter As defined in Admob/DFP console
   * @param mediationAdRequest Additional ad data
   * @param customEventExtras Extra data can be provided
   */
  @Override
  public void requestInterstitialAd(
      Context context,
      CustomEventInterstitialListener customEventListener,
      String serverParameter,
      com.google.android.gms.ads.mediation.MediationAdRequest mediationAdRequest,
      Bundle customEventExtras) {

    // You must set the mediation name to either AdMob or DFP
    // MEDIATOR_NAME = InneractiveMediationName.ADMOB;
    // Or
    // MEDIATOR_NAME = InneractiveMediationName.DFP;
    if (MEDIATOR_NAME == null || (MEDIATOR_NAME != InneractiveMediationName.ADMOB
        && MEDIATOR_NAME != InneractiveMediationName.DFP)) {
      Log.i(TAG, InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL
              + " - you must set mediation name");
      customEventListener.onAdFailedToLoad(AdRequest.ERROR_CODE_INVALID_REQUEST);
      return;
    }

    String resultSpotId = null;
    String resultKeywords = null;
    //DFP note - parameter string in the DFP console needs to be escaped, or else it will break
    //the google dfp parser. escaped example - {\"key\":\"value\"}
    //Escaping the JSON is not needed for admob server parameter.
    JSONObject serverExtras = null;
    if (serverParameter != null) {
      try {
        serverExtras = new JSONObject(serverParameter);
      } catch (Exception ignored) {
      }
    }

    if (serverExtras != null) {
      resultSpotId = serverExtras.optString(InneractiveMediationDefs.REMOTE_KEY_SPOT_ID);
      resultKeywords = serverExtras.optString(InneractiveMediationDefs.REMOTE_KEY_KEYWORDS);
    }

    //Make sure we have a spotId from the request.
    if (TextUtils.isEmpty(resultSpotId)) {
      resultSpotId = SAMPLE_INTERSTITIAL_SPOT_ID;
      if (TextUtils.isEmpty(resultSpotId)) {
        customEventListener.onAdFailedToLoad(AdRequest.ERROR_CODE_INVALID_REQUEST);
        return;
      }
    }

    mInterstitialListener = customEventListener;
    mContext = context;

    StringBuilder iaKeyWords = new StringBuilder();
    String iaZip = null;

    // Check whether the Ad Request contains user data which can be added to Inneractive Request
    if (mediationAdRequest != null) {
      Set<String> keyWords = mediationAdRequest.getKeywords();
      if (keyWords != null) {
        for (String key : keyWords) {
          if (iaKeyWords.length() > 0) {
            iaKeyWords.append(",");
          }

          iaKeyWords.append(key);
        }
      }
    }

    if (customEventExtras != null) {
      iaZip = customEventExtras.getString(InneractiveMediationDefs.KEY_ZIPCODE);
    }

    if (mInterstitialSpot != null) {
      mInterstitialSpot.destroy();
    }

    mInterstitialSpot = InneractiveAdSpotManager.get().createSpot();
    // Set your mediation name
    mInterstitialSpot.setMediationName(MEDIATOR_NAME);

    InneractiveFullscreenUnitController fullscreenUnitController = new InneractiveFullscreenUnitController();
    mInterstitialSpot.addUnitController(fullscreenUnitController);

    InneractiveAdRequest request = new InneractiveAdRequest(resultSpotId);
    request.setUserParams(new InneractiveUserConfig()
        .setZipCode(iaZip));

    if (!TextUtils.isEmpty(resultKeywords)) {
			if (iaKeyWords.length() > 0) {
				iaKeyWords.append(",");
			}
      iaKeyWords.append(resultKeywords);
    }

    if (iaKeyWords.length() > 0) {
      request.setKeywords(iaKeyWords.toString());
    }

    mInterstitialSpot.setRequestListener(new InneractiveAdSpot.RequestListener() {

      /**
       * Called by Inneractive when an interstitial is ready for display
       * @param adSpot Spot object
       */
      @Override
      public void onInneractiveSuccessfulAdRequest(InneractiveAdSpot adSpot) {
        Log.i(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL,
            "Interstitial loaded successfully!");
        mInterstitialListener.onAdLoaded();
      }

      /**
       * Called by Inneractive an interstitial fails loading
       * @param adSpot Spot object
       * @param errorCode the failure's error.
       */
      @Override
      public void onInneractiveFailedAdRequest(InneractiveAdSpot adSpot,
          InneractiveErrorCode errorCode) {
        Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL,
            "Failed loading interstitial! with error: " + errorCode);
        if (errorCode == InneractiveErrorCode.CONNECTION_ERROR
            || errorCode == InneractiveErrorCode.CONNECTION_TIMEOUT) {
          mInterstitialListener.onAdFailedToLoad(AdRequest.ERROR_CODE_NETWORK_ERROR);
        } else if (errorCode == InneractiveErrorCode.NO_FILL) {
          mInterstitialListener.onAdFailedToLoad(AdRequest.ERROR_CODE_NO_FILL);
        } else {
          mInterstitialListener.onAdFailedToLoad(AdRequest.ERROR_CODE_INTERNAL_ERROR);
        }
      }
    });

    mInterstitialSpot.requestAd(request);
  }

  /**
   * Called by the Admob/DFP infra-structure in order for the plugin to start showing Inneractive's
   * interstitial
   */
  @Override
  public void showInterstitial() {
    // check if the ad is ready
    if (mInterstitialSpot != null && mInterstitialSpot.isReady()) {
      InneractiveFullscreenUnitController fullscreenUnitController = (InneractiveFullscreenUnitController) mInterstitialSpot
          .getSelectedUnitController();
      fullscreenUnitController.setEventsListener(new InneractiveFullscreenAdEventsListener() {

        /**
         * Called by Inneractive when an interstitial ad activity is closed
         * @param adSpot Spot object
         */
        @Override
        public void onAdDismissed(InneractiveAdSpot adSpot) {
          mInterstitialListener.onAdClosed();
          Log.i(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "onAdDismissed");
        }

        /**
         * Called by Inneractive when an interstitial ad activity is shown
         * @param adSpot Spot object
         */
        @Override
        public void onAdImpression(InneractiveAdSpot adSpot) {
          mInterstitialListener.onAdOpened();
          Log.i(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "onAdImpression");
        }

        /**
         * Called by Inneractive when an interstitial ad is clicked
         * @param adSpot Spot object
         */
        @Override
        public void onAdClicked(InneractiveAdSpot adSpot) {
          mInterstitialListener.onAdClicked();
          Log.i(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "onAdClicked");
        }

        /**
         * Called by Inneractive when an interstitial ad opened an external application
         * @param adSpot Spot object
         */
        @Override
        public void onAdWillOpenExternalApp(InneractiveAdSpot adSpot) {
          mInterstitialListener.onAdLeftApplication();
          Log.i(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "onAdWillOpenExternalApp");
        }

        /**
         * Called when an ad has entered an error state, this will only happen when the ad is being shown
         *
         * @param adSpot the relevant ad spot
         */
        @Override
        public void onAdEnteredErrorState(InneractiveAdSpot adSpot, AdDisplayError error) {
          Log.i(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL, "onAdEnteredErrorState - " + error.getMessage());
        }

        /**
         * Called by Inneractive when Inneractive's internal browser, which was opened by this interstitial, was closed
         * @param adSpot Spot object
         */
        @Override
        public void onAdWillCloseInternalBrowser(InneractiveAdSpot adSpot) {
          Log.i(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL,
              "onAdWillCloseInternalBrowser");
        }
      });

      // Add video content controller, for controlling video ads
      InneractiveFullscreenVideoContentController videoContentController = new InneractiveFullscreenVideoContentController();
      videoContentController.setEventsListener(new VideoContentListener() {
        @Override
        public void onProgress(int totalDurationInMsec, int positionInMsec) {
          Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL,
              "Interstitial: Got video content progress: total time = " + totalDurationInMsec
                  + " position = " + positionInMsec);
        }

        /**
         * Called by inneractive when an Intersititial video ad was played to the end
         * <br>Can be used for incentive flow
         * <br>Note: This event does not indicate that the interstitial was closed
         */
        @Override
        public void onCompleted() {
          Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL,
              "Interstitial: Got video content completed event");
        }

        @Override
        public void onPlayerError() {
          Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL,
              "Interstitial: Got video content play error event");
        }
      });

      // Now add the content controller to the unit controller
      fullscreenUnitController.addContentController(videoContentController);

      fullscreenUnitController.show((Activity) mContext);
    } else {
      Log.d(InneractiveMediationDefs.IA_LOG_FOR_ADMOB_INTERSTITIAL,
          "The Interstitial ad is not ready yet.");
    }
  }

  /**
   * Called by Admob/DFP when an ad should be destroyed
   * <br>Destroy the underline Inneractive ad
   */
  @Override
  public void onDestroy() {
    if (mInterstitialSpot != null) {
      mInterstitialSpot.destroy();
      mInterstitialSpot = null;
    }
  }

  /**
   * Called by Admob/DFP when an ad should be paused. Usually due to the Activity:onPause event
   * <br>Can be used for example to stop a video ad
   */
  @Override
  public void onPause() {
    // Currently not implemeted
  }

  /**
   * Called by Admob/DFP when an ad should be resumed. Usually due to the Activity:onResume event
   * <br>Can be used for example to resume a video ad
   */
  @Override
  public void onResume() {
    // Currently not implemeted
  }
}
