package io.g740.client.service.impl;

import io.g740.client.dto.SQLGenerResultDTO;
import io.g740.client.entity.DataExportTaskDO;
import io.g740.client.entity.DfFormTableSettingDO;
import io.g740.client.executor.Executor;
import io.g740.client.executor.impl.CommonExportExecutor;
import io.g740.client.executor.impl.ExecutorBuilder;
import io.g740.client.service.DataExportService;
import io.g740.client.service.QueryDataSercive;
import io.g740.client.util.DateUtils;
import io.g740.client.util.FileUtils;
import io.g740.client.util.HttpClientUtil;
import io.g740.client.util.StringUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.PropertyNamingStrategy;
import com.alibaba.fastjson.serializer.SerializeConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletResponse;
import javax.sql.DataSource;
import java.io.*;
import java.nio.file.Paths;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * @function:
 * @author: DAM
 * @date: 2019/7/9 14:58
 * @description:
 * @version: V1.0
 */
@Service
public class DataExportServiceImpl implements DataExportService {

    private static final Logger LOGGER = LoggerFactory.getLogger(DataExportServiceImpl.class);

    private ExecutorService executorService = Executors.newFixedThreadPool(1);

    @Autowired
    private QueryDataSercive queryDataSercive;


    @Override
    public DataExportTaskDO findDataExportTaskById(Long taskId, String url) throws Exception {
        Map<String, String[]> simpleParameters = new HashMap<>();
        simpleParameters.put("taskId", new String[]{taskId + ""});
        String result = HttpClientUtil.doGet(url, simpleParameters);
        DataExportTaskDO dataExportTaskDO = JSONObject.parseObject(result, DataExportTaskDO.class);
        return dataExportTaskDO;
    }

    @Override
    public DataExportTaskDO addDataExportTask(DataExportTaskDO dataExportTask, String exportAddTaskUrl) throws Exception {
        String jsonParams = getJsonString(dataExportTask);
        String result = HttpClientUtil.doPostJson(exportAddTaskUrl, jsonParams, null);
        DataExportTaskDO dataExportTaskDO = JSONObject.parseObject(result, DataExportTaskDO.class);
        return dataExportTaskDO;
    }

    @Override
    public DataExportTaskDO updateDataExportTask(DataExportTaskDO dataExportTaskDO, String exportUpdateTaskUrl) throws Exception {
        //驼峰转划线
        String jsonParams = getJsonString(dataExportTaskDO);
        String result = HttpClientUtil.doPostJson(exportUpdateTaskUrl, jsonParams, null);
        DataExportTaskDO dataExportTask = JSONObject.parseObject(result, DataExportTaskDO.class);
        return dataExportTask;
    }


    @Override
    public List<DfFormTableSettingDO> getAllDsFormTableSettingByDfKeyForExport(String dataFacetKey, String url) {
        Map<String, String[]> simpleParameters = new HashMap<String, String[]>();
        simpleParameters.put("data_facet_key", new String[]{dataFacetKey});
        String result = HttpClientUtil.doGet(url, simpleParameters);
        List<DfFormTableSettingDO> DfFormTableSettingDOS = JSONObject.parseArray(result, DfFormTableSettingDO.class);
        return DfFormTableSettingDOS;
    }


    @Override
    public Long executeExportTask(String dataFacetKey,
                                  DataSource dataSource,
                                  String fileName,
                                  String fileTempPath,
                                  String exportAddTaskUrl,
                                  String exportUpdateTaskUrl,
                                  String formTableSettingUrl,
                                  String formTableSettingForExportUrl,
                                  Map<String, String[]> requestParams) throws Exception {
        if (StringUtils.isNullOrEmpty(dataFacetKey)) {
            throw new Exception("Empty data source key " + dataFacetKey);
        }
        //参数配置
        DataExportTaskDO dataExportTask = new DataExportTaskDO();
        String now = DateUtils.ofLongStr(new Date());
        dataExportTask.setStartAt(now);
        dataExportTask.setFileName(fileName);

        long start = System.currentTimeMillis();
        LOGGER.info("async export 开始调用data export task");
        final DataExportTaskDO toWaitSaveExportTask = addDataExportTask(dataExportTask, exportAddTaskUrl);
        LOGGER.info("async export 开始调用data export task:{}", (System.currentTimeMillis() - start));
        start = System.currentTimeMillis();
        this.executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    //生成导出文件
                    File exportFile = export(dataFacetKey,dataSource,
                            fileTempPath,
                            formTableSettingUrl,
                            formTableSettingForExportUrl,
                            dataExportTask, requestParams);

                    toWaitSaveExportTask.setFileName(exportFile.getName());
                    toWaitSaveExportTask.setFilePath(exportFile.getAbsolutePath());
                    toWaitSaveExportTask.setEndAt(DateUtils.ofLongStr(new Date()));

                    updateDataExportTask(toWaitSaveExportTask,exportUpdateTaskUrl );
                } catch (Exception e) {
                    LOGGER.error("导出文件失败,id: {}", dataExportTask.getId(), e);

                    toWaitSaveExportTask.setFailedAt(DateUtils.ofLongStr(new Date()));
                    toWaitSaveExportTask.setDetails(e.getMessage());
                    try {

                        updateDataExportTask(toWaitSaveExportTask,exportUpdateTaskUrl);
                    } catch (Exception e1) {
                        LOGGER.info("Save user export task failure");
                    }
                }
            }
        });
        return toWaitSaveExportTask.getId();
    }


    @Override
    public File export(String dataFacetKey,
                       DataSource dataSource,
                       String fileTempPath,
                       String formTableSettingUrl,
                       String formTableSettingForExportUrl,
                       DataExportTaskDO dataExportTaskDO,
                       Map<String, String[]> requestParams) throws Exception {
        String fileName = dataExportTaskDO.getFileName();
        String fullFilePathOfExportFile = FileUtils.contact(fileTempPath, fileName);
        //临时文件处理
        File tmpFile=new File(fullFilePathOfExportFile);
        File parentFile = tmpFile.getParentFile();
        // 如果没有文件夹则创建
        if (!parentFile.exists()) {
            parentFile.mkdirs();
        }
        //获取生成sql文件
        SQLGenerResultDTO sqlGenerResultDTO = queryDataSercive.generalQuery(dataFacetKey,
                false,
                formTableSettingUrl,
                requestParams);
        if(sqlGenerResultDTO==null){
            throw  new Exception("get general sql failed");
        }
        List<DfFormTableSettingDO> queryTableSettings = getAllDsFormTableSettingByDfKeyForExport(dataFacetKey, formTableSettingForExportUrl);

        String querySql = sqlGenerResultDTO.getQuerySql();
        List<Object> paramList=sqlGenerResultDTO.getParamsValue();

        Executor build = ExecutorBuilder.getInstance().dataSource(dataSource).exportExecutor(new CommonExportExecutor()).build();

        return build.exportExcel(querySql, paramList,queryTableSettings, Paths.get(fullFilePathOfExportFile));
    }


    /**
     * 驼峰转下划线
     *
     * @param o
     * @return
     */
    private String getJsonString(Object o) {
        SerializeConfig config = new SerializeConfig();
        config.propertyNamingStrategy = PropertyNamingStrategy.SnakeCase;
        return JSON.toJSONString(o, config);
    }


    @Override
    public void exportFileDownload(Long taskId, String exportTaskStatusUrl, HttpServletResponse response) throws Exception {
        DataExportTaskDO dataExportTaskDO = findDataExportTaskById(taskId, exportTaskStatusUrl);
        if (dataExportTaskDO == null) {
            throw new Exception("taskId is not found");
        }

        if(StringUtils.isNotNullNorEmpty(dataExportTaskDO.getFailedAt())||StringUtils.isNullOrEmpty(dataExportTaskDO.getFilePath())){
            throw new Exception("export file is failed");
        }

        String filePath = dataExportTaskDO.getFilePath();
        File file = new File(filePath);
        response.setCharacterEncoding("UTF-8");
        response.setHeader("content-type", "application/octet-stream;charset=UTF-8");
        response.setContentType("application/octet-stream;charset=UTF-8");
        //加上设置大小下载下来的.xlsx文件打开时才不会报“Excel 已完成文件级验证和修复。此工作簿的某些部分可能已被修复或丢弃”
        response.addHeader("Content-Length", String.valueOf(new FileInputStream(file).available()));
        try {
            response.setHeader("Content-Disposition", "attachment;filename=" + java.net.URLEncoder.encode(file.getName(), "UTF-8"));
        } catch (UnsupportedEncodingException e) {
            System.out.println("error:" + e);
        }
        byte[] buff = new byte[1024];
        BufferedInputStream bis = null;
        OutputStream os = null;
        try {
            os = response.getOutputStream();
            bis = new BufferedInputStream(new FileInputStream(file));
            int i = bis.read(buff);
            while (i != -1) {
                os.write(buff, 0, buff.length);
                os.flush();
                i = bis.read(buff);
            }
            LOGGER.info("下载成功,filePath=", filePath);
        } catch (IOException e) {
            LOGGER.error("下载失败,filePath={}", filePath);
        } finally {
            if (bis != null) {
                try {
                    bis.close();
                } catch (IOException e) {
                    LOGGER.error("error:", e);
                }
            }
            if (os != null) {
                try {
                    os.close();
                    os.flush();
                } catch (IOException e) {
                    LOGGER.error("error:", e);
                }
            }
        }
    }


}
