package com.fxlabs;

/*
 * Copyright 2001-2005 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.springframework.http.*;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.client.RestTemplate;

import java.util.Arrays;
import java.util.concurrent.TimeUnit;

/**
 * @author Mohammed Shoukath Ali
 */

@Mojo(name = "job")
public class JobMojo
        extends AbstractMojo {


    @Parameter(property = "job.message", defaultValue = "Testing Job")
    private String message;

    @Parameter(property = "job.jobId")
    private String jobId;

//    @Parameter( property = "job.url", defaultValue = "https://cloud.fxlabs.io")
//    private String url;


    @Parameter(property = "job.region")
    private String region;

    @Parameter(property = "job.host", defaultValue = HOST_URL)
    private String host;

    @Parameter(property = "job.username")
    private String username;

    @Parameter(property = "job.password")
    private String password;

    private static final String HOST_URL = "https://cloud.fxlabs.io";

    private static final String UAT_HOST_URL = "http://13.56.210.25";

    private static final String LOCAL_HOST_URL = "http://localhost:8080";

    private static final String RUN_JOB_API_ENDPOINT = "/api/v1/runs/job/";

    private static final String RUN_STATUS_JOB_API_ENDPOINT = "/api/v1/runs/";


    public void execute() throws MojoExecutionException {
        try {
            getLog().info("Username : " + username);
            getLog().info("Job Id : " + jobId);
            getLog().info("Region : " + region);

            String hostUrl = HOST_URL;

            String jobRunUrl = host + RUN_JOB_API_ENDPOINT + jobId + "?region=" + region;
            String runStatusUrl = host + RUN_STATUS_JOB_API_ENDPOINT;

            RestTemplate restTemplate = new RestTemplate(HttpClientFactoryUtil.httpComponentsClientHttpRequestFactory());
            HttpHeaders httpHeaders = new HttpHeaders();

            httpHeaders.setContentType(MediaType.APPLICATION_JSON);
            httpHeaders.set("Accept", "application/json");


            httpHeaders.set("Authorization", AuthBuilder.createBasicAuth(username, password));

            HttpEntity<String> request = new HttpEntity<>("", httpHeaders);

            ResponseEntity<String> response = null;
            int statusCode = -1;
            String responseBody = null;
            HttpHeaders headers = null;

            getLog().info("calling " + jobRunUrl);
            response = restTemplate.exchange(jobRunUrl, HttpMethod.POST, request, String.class);

            String runId = getByKeyId(response.getBody(), "id", "");

            if (StringUtils.isEmpty(runId)) {
                throw new MojoExecutionException("Invalid runid." + runId);
            }
            getLog().info("Run Id : " + runId);

            String status = getByKeyId(response.getBody(), "task", "status");

            if (StringUtils.isEmpty(status)) {
                throw new MojoExecutionException("Invalid status " + status);
            }

            getLog().info("status : " + status);

            do {
                try {
                    //getLog().info("Waiting for 5 secs ........");
                    TimeUnit.SECONDS.sleep(5);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }

                getLog().info("Checking status ........");
                response = restTemplate.exchange(runStatusUrl + runId, HttpMethod.GET, request, String.class);
                String ciCdStatus = getByKeyId(response.getBody(), "ciCdStatus", "");

                if (StringUtils.isEmpty(ciCdStatus)) {
                    throw new MojoExecutionException("Failed to get status.");
                }

                String[] dataArray = ciCdStatus.split(":");

                status = dataArray[0];

                if (StringUtils.isEmpty(status)) {
                    throw new MojoExecutionException("Invalid status." + status);
                }

                getLog().info("Run Status: " + status + ".....");

                if ("COMPLETED".equalsIgnoreCase(status)) {
                    printStatus(dataArray);
                }

                if ("FAIL".equalsIgnoreCase(status)) {

                    getLog().info("Reason for Failure : " + dataArray[1]);
                }


                if ("TIMEOUT".equalsIgnoreCase(status)) {
                    printStatus(dataArray);

                    throw new MojoExecutionException("Job timeout.....");

                }


            } while ("WAITING".equalsIgnoreCase(status) || "PROCESSING".equalsIgnoreCase(status));

        } catch (HttpStatusCodeException statusCodeException) {
            getLog().info(statusCodeException.toString() + statusCodeException.getResponseBodyAsString() + statusCodeException.getResponseHeaders());
            throw new MojoExecutionException(statusCodeException.toString() + statusCodeException.getResponseBodyAsString() + statusCodeException.getResponseHeaders());
        } catch (Exception e) {
            getLog().info(e.getLocalizedMessage());
            throw new MojoExecutionException(e.getLocalizedMessage());
        }

    }

    private void printStatus(String[] dataArray) {

        getLog().info("----------------------------------------");
        getLog().info("Run detail's link " + host + dataArray[6]);
        getLog().info("----------------------------------------");
        getLog().info(dataArray[7]);
        getLog().info("----------------------------------------");
        getLog().info("Run No : " + dataArray[5]);
        getLog().info("----------------------------------------");
        getLog().info("Success : " + dataArray[1] + " % " + " -- Total Tests : " + dataArray[2] + " -- Failed Tests : " + dataArray[3]);
        getLog().info("----------------------------------------");
    }


    public String getByKeyId(String value, String key1, String key2) {

        if (StringUtils.isEmpty(value)) {
            return null;
        }

        ObjectMapper mapper = new ObjectMapper();
        //  jsonMapper.configure(DeserializationConfig.ACCEPT_EMPTY_STRING_AS_NULL_OBJECT, true)

        try {
            JsonNode rootNode = mapper.readTree(value);

            JsonNode nameNode = null;

            if (StringUtils.isEmpty(key2) && !StringUtils.isEmpty(key1)) {
                nameNode = rootNode.path("data").path(key1);
            } else if (!StringUtils.isEmpty(key2) && !StringUtils.isEmpty(key1)) {
                nameNode = rootNode.path("data").path(key1).path(key2);
            }

            return nameNode.textValue();
        } catch (Exception e) {
            getLog().info(e.getLocalizedMessage());
        }
        return null;
    }

}
