package io.fogcloud.fog_mqtt.service;

import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;

import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.MqttCallback;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.eclipse.paho.client.mqttv3.MqttPersistenceException;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;

import android.app.Service;
import android.content.Intent;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;

import io.fogcloud.fog_mqtt.helper.ComHelper;
import io.fogcloud.fog_mqtt.helper.ConstParams;

public class MqttService extends Service {
	private ServiceBinder serviceBinder = new ServiceBinder();
	// private String LOG_TAG = "---service---";

	private MqttClient client;
	private MqttConnectOptions options;
	private Handler handler;
	private ScheduledExecutorService scheduler;
	private MqttServiceListener mMqttServiceListener = null;
	private Boolean recvTag = false;
	private Boolean connectTag = false;

	private String[] topicList = null;

	@Override
	public IBinder onBind(Intent intent) {
		// Log.d(LOG_TAG, "onBind");

		String host = intent.getStringExtra("com.mxchip.host");
		String port = intent.getStringExtra("com.mxchip.port");
		String userName = intent.getStringExtra("com.mxchip.userName");
		String passWord = intent.getStringExtra("com.mxchip.passWord");
		String clientID = intent.getStringExtra("com.mxchip.clientID");
		String topic = intent.getStringExtra("com.mxchip.topic");
		boolean isencrypt = intent.getBooleanExtra("com.mxchip.isencrypt", false);

		startMqttService(host, port, userName, passWord, clientID, topic, isencrypt);

		return serviceBinder;
	}

	@Override
	public void onCreate() {
		// Log.d(LOG_TAG, "onCreate");
		super.onCreate();
	}

	@Override
	public void onDestroy() {
		// Log.d(LOG_TAG, "onDestroy");
		sendMsgToClient(ConstParams._STOP_CODE, ConstParams._STOP_MSG);
		stopMqttService();

		super.onDestroy();
	}

	/**
	 * start mqtt service
	 * @param host host
	 * @param port port
	 * @param userName userName
	 * @param password password
	 * @param clientID clientID
	 * @param topic  topic
	 * @param isencrypt is need secrety
     */
	public void startMqttService(String host, String port, String userName,
			String password, String clientID, final String topic, boolean isencrypt) {
		
		//mybe it without port
		String URI = "";
		if(ComHelper.checkPara(port))
			URI = host + ":" + port;
		else
			URI = host;
		
		if(isencrypt){
			mqttInit("ssl://" + URI, userName, password, clientID, true);
		}else{
			mqttInit("tcp://" + URI, userName, password, clientID, false);
		}
		handler = new Handler() {
			@Override
			public void handleMessage(Message msg) {
				super.handleMessage(msg);
				if (msg.what == 1) {// message arrived
					// Log.d("---ok---", (String) msg.obj);
					sendMsgToClient(ConstParams._PAYLOAD_CODE, (String) msg.obj);
				} else if (msg.what == 2) {// connection
					connectTag = true;
					reSubscribe(topic);
					// Log.d(LOG_TAG, "Connected");
					sendMsgToClient(ConstParams._CON_CODE, ConstParams._CON_MSG);
				} else if (msg.what == 3) {// connect exception
				// Log.d(LOG_TAG, "Connect Exception");
					sendMsgToClient(ConstParams._EXCEP_CODE,
							ConstParams._EXCEP_MSG);
				}
			}
		};
		startReconnect();
	}

	/**
	 * subscibe again
	 * @param topic topic
     */
	private void reSubscribe(String topic) {
		if (!topic.equals("") && (null == topicList)) {
			try {
				client.subscribe(topic, 0);
			} catch (Exception e) {
				e.printStackTrace();
			}
			topicList = new String[1];
			topicList[0] = topic;
		} else {
			for (String tp : topicList) {
				try {
					client.subscribe(tp, 0);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
		sendMsgToClient(ConstParams._RESUB_CODE, ConstParams._RESUB_MSG);
	}

	/**
	 * init mqtt
	 * @param host host
	 * @param userName userName
	 * @param password password
	 * @param clientID clientID
     * @param isencrypt is need
     */
	private void mqttInit(String host, String userName, String password, String clientID, boolean isencrypt) {
		try {
			/*
			 * host client id must different MemoryPersistence is default
			 */
			client = new MqttClient(host, clientID, new MemoryPersistence());
			// MQTT setting
			options = new MqttConnectOptions();
			/*
			 * if clean session false, the service will remember the id of
			 * client true, it will create a new id when it connected
			 */
			options.setCleanSession(true);
			/*
			 * user name
			 */
			options.setUserName(userName);
			/*
			 * password
			 */
			options.setPassword(password.toCharArray());
			/*
			 * overtime 10ms
			 */
			options.setConnectionTimeout(10);
			/*
			 * HeartBeat service will send heart beat once 1.5*20 but it doesn't
			 * reconnect
			 */
			// options.setKeepAliveInterval(20);

			if (isencrypt) {
				//创建SSL连接
				try {
					TrustManagerFactory tmf = TrustManagerFactory.getInstance("X509");
					tmf.init((KeyStore) null);
					TrustManager[] trustManagers = tmf.getTrustManagers();

//					创建返利授权证书
					SSLContext sslc = SSLContext.getInstance("SSLv3");

					sslc.init(null, trustManagers, null);

//					配置mqtt连接
					options.setSocketFactory(sslc.getSocketFactory());
				} catch (NoSuchAlgorithmException e) {
					e.printStackTrace();
				} catch (KeyManagementException e) {
					e.printStackTrace();
				} catch (KeyStoreException e) {
					e.printStackTrace();
				}
			}

			// call back
			client.setCallback(new MqttCallback() {

				@Override
				public void connectionLost(Throwable cause) {
					/**
					 * if it lost, we wil reconnect
					 */
					connectTag = false;
					// Log.d(LOG_TAG, "Lost");
					sendMsgToClient(ConstParams._LOST_CODE,
							ConstParams._LOST_MSG);
				}

				@Override
				public void deliveryComplete(IMqttDeliveryToken token) {
					/*
					 * after publish we will use it
					 */
					sendMsgToClient(ConstParams._PUB_CODE, ConstParams._PUB_MSG);
				}

				@Override
				public void messageArrived(String topicName, MqttMessage payload)
						throws Exception {
					// after subscribe we will receive
					if (recvTag) {
						Message msg = new Message();
						msg.what = 1;
						msg.obj = "{\"topic\":\"" + topicName
								+ "\",\"payload\":" + payload.toString() + "}";
						handler.sendMessage(msg);
					}
				}
			});
		} catch (MqttException e) {
			e.printStackTrace();
		}
	}

	/**
	 * subscribe topic
	 * @param topic topic
	 * @param qos 012
     */
	public void subscribeService(String topic, int qos) {
		if (topic.equals("")) {
			sendMsgToClient(ConstParams._EMPTY_CODE, ConstParams._EMPTY_MSG);
		} else if (!connectTag) {
			sendMsgToClient(ConstParams._DISCON_CODE, ConstParams._DISCON_MSG);
		} else {
			try {
				client.subscribe(topic, qos);

				int ssize = topicList.length;
				String[] tmp = new String[ssize + 1];
				int i = 0;
				for (; i < ssize; i++) {
					tmp[i] = topicList[i];
				}
				tmp[i] = topic;

				topicList = tmp;
			} catch (Exception e) {
				e.printStackTrace();
			}
			sendMsgToClient(ConstParams._SUB_CODE, ConstParams._SUB_MSG);
		}
	}

	/**
	 * cancel subscribe topic
	 * @param topic topic
     */
	public void unSubscribeService(String topic) {
		if (topic.equals("")) {
			sendMsgToClient(ConstParams._EMPTY_CODE, ConstParams._EMPTY_MSG);
		} else if (!connectTag) {
			sendMsgToClient(ConstParams._DISCON_CODE, ConstParams._DISCON_MSG);
		} else {
			try {
				client.unsubscribe(topic);

				int ssize = topicList.length;
				int i = 0, j = 0;
				for (; i < ssize; i++) {
					if (topic != topicList[i]) {
						j++;
					}
				}
				String[] tmp = new String[j];
				i = 0;
				j = 0;
				for (; i < ssize; i++) {
					if (topic != topicList[i]) {
						tmp[j++] = topicList[i];
					}
				}
				topicList = tmp;

			} catch (Exception e) {
				e.printStackTrace();
			}
			sendMsgToClient(ConstParams._UNSUB_CODE, ConstParams._UNSUB_MSG);
		}
	}

	/**
	 * if the connect is lost, it will reconnect
	 */
	private void startReconnect() {
		scheduler = Executors.newSingleThreadScheduledExecutor();
		scheduler.scheduleAtFixedRate(new Runnable() {

			@Override
			public void run() {
				if (!client.isConnected()) {
					connect();
				}
			}
		}, 0 * 1000, 10 * 1000, TimeUnit.MILLISECONDS);
	}

	/**
	 * connect the mqtt service
	 */
	private void connect() {
		new Thread(new Runnable() {

			@Override
			public void run() {
				try {
					client.connect(options);
					Message msg = new Message();
					msg.what = 2;
					handler.sendMessage(msg);
				} catch (Exception e) {
					// e.printStackTrace();
					Message msg = new Message();
					msg.what = 3;
					handler.sendMessage(msg);
				}
			}
		}).start();
	}

	/**
	 * start receive messages
	 * @param mqttServiceListener call back
     */
	public void initMessageSend(MqttServiceListener mqttServiceListener) {
		mMqttServiceListener = mqttServiceListener;
		recvTag = true;
	}

	public void recvMsgService() {
		recvTag = true;
	}

	/**
	 * stop receive messages
	 */
	public void stopRecvMsgService() {
		recvTag = false;
	}

	/**
	 * publish command to service
	 * @param topic topic
	 * @param command command
	 * @param qos qos
	 * @param retained retained
     */
	public void publishService(String topic, String command, int qos,
			boolean retained) {
		try {
			client.publish(topic, command.getBytes(), qos, retained);
		} catch (MqttPersistenceException e) {
			e.printStackTrace();
		} catch (MqttException e) {
			e.printStackTrace();
		}
	}

	/**
	 * stop mqtt service
	 */
	public void stopMqttService() {
		try {
			scheduler.shutdown();
			client.disconnect();
			connectTag = false;
			recvTag = false;
		} catch (MqttException e) {
			e.printStackTrace();
		}
	}

	/**
	 *
	 * @param code code
	 * @param message message
     */
	private void sendMsgToClient(int code, String message) {
		if (null != mMqttServiceListener) {
			mMqttServiceListener.onMqttReceiver(code, message);
		}
	}

	/*
	 * ---------------------------------------end mqtt
	 */

	/*
	 * ---------------------------------------begin binder
	 */
	public class ServiceBinder extends android.os.Binder {
		public MqttService getService() {
			return MqttService.this;
		}

		/**
		 * bind listener then we can send message to api
		 * 
		 * @param mqttServiceListener callback
		 */
		public void bindListenerAPI(MqttServiceListener mqttServiceListener) {
			initMessageSend(mqttServiceListener);
		}

		/**
		 * stop sending message to api
		 */
		public void stopRecvMsgAPI() {
			stopRecvMsgService();
		}

		/**
		 * publish from api
		 * 
		 * @param topic topic
		 * @param command command
		 * @param qos 012
		 * @param retained 0
		 */
		public void publishAPI(String topic, String command, int qos,
				boolean retained) {
			publishService(topic, command, qos, retained);
		}

		/**
		 * add subscribe from api
		 * 
		 * @param topic topic
		 * @param qos 012
		 */
		public void addSubscribeAPI(String topic, int qos) {
			subscribeService(topic, qos);
		}

		/**
		 * remove one topic of topics by api
		 * 
		 * @param topic topic
		 */
		public void unSubscribeAPI(String topic) {
			unSubscribeService(topic);
		}

		/***
		 * want receive message again
		 */
		public void recvMsgAPI() {
			recvMsgService();
		}
	}
	/*
	 * --------------------------------------end binder
	 */
}
