package io.featureflow.client;



import com.google.gson.JsonPrimitive;
import io.featureflow.client.core.*;
import io.featureflow.client.core.CallbackEvent;
import io.featureflow.client.model.Event;
import io.featureflow.client.model.Feature;
import io.featureflow.client.model.FeatureControl;
import io.featureflow.client.model.Variant;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Closeable;
import java.io.IOException;
import java.time.LocalTime;
import java.util.*;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;


/**
 * Created by oliver on 23/05/2016.
 * The featureflow client is the interface to clients using featureflow.
 * The client uses the SSE Stream, Rest Client and Repository to manage features
 * Configuration may be provided to register feature controls and capabilitites
 * Callbacks may be assigned to proved ondemand updates to feature control changes
 */
public class FeatureflowClient implements Closeable{


    private static final Logger logger = LoggerFactory.getLogger(FeatureflowClient.class);
    private final FeatureflowConfig config;
    private final FeatureControlStreamClient featureControlStreamClient; // manages pubsub events to update a feature control
    private final FeatureControlCache featureControlCache; //holds the featureControls
    private final RestClient restClient; //manages retrieving features and pushing updates
    private final FeatureEventHandler eventHandler;
    private final Map<String, Feature> featuresMap = new HashMap<>(); //this contains code registered features and failovers
    private Queue<FeatureControlCallbackHandler> handlers;

    FeatureflowClient(String apiKey, List<Feature> features, FeatureflowConfig config, Map<CallbackEvent, List<FeatureControlCallbackHandler>> callbacks) {
        //set config, use a builder
        this.config = config;

        featureControlCache = new SimpleMemoryFeatureCache();
        restClient = new RestClient(apiKey, config);
        //featureControlEventHandler = new FeatureControlEventHandler(restClient);
        eventHandler = new FeatureEventHandler(config, restClient);

        //Actively defining registrations helps alert if features are available in an environment
        if(features !=null&& features.size()>0){
            for (Feature feature : features) {
                featuresMap.put(feature.key, feature);
            }
            try {
                restClient.registerFeatureControls(features);
            } catch (IOException e) {
                logger.error("Problem registering feature controls", e);
            }
        }

        featureControlStreamClient = new FeatureControlStreamClient(apiKey, config, featureControlCache, callbacks);
        Future<Void> startFuture = featureControlStreamClient.start();
        if (config.waitForStartup > 0L) {
            logger.info("Waiting for Featureflow to inititalise");
            try {
                startFuture.get(config.waitForStartup, TimeUnit.MILLISECONDS);
            } catch (TimeoutException e) {
                logger.error("Timeout waiting for Featureflow client initialise");
            } catch (Exception e) {
                logger.error("Exception waiting for Featureflow client to initialise", e);
            }
        }
    }

    public Evaluate evaluate(String featureKey, FeatureflowContext featureflowContext) {
        Evaluate e = new Evaluate(this, featureKey, featureflowContext);
        return e;

    }

    public Evaluate evaluate(String featureKey) {
        //create and empty context
        FeatureflowContext featureflowContext = FeatureflowContext.context().build();
        return evaluate(featureKey, featureflowContext);
    }

    public Map<String, String> evaluateAll(FeatureflowContext featureflowContext){
        Map<String, String> result = new HashMap<>();
        for(String s: featureControlCache.getAll().keySet()){
            result.put(s, eval(s, featureflowContext));
        }
        return result;
    }

    private String eval(String featureKey, FeatureflowContext featureflowContext) {
        String failoverVariant = (featuresMap.get(featureKey)!=null&&featuresMap.get(featureKey).failoverVariant!=null)?featuresMap.get(featureKey).failoverVariant: Variant.off;
        FeatureControl control = featureControlCache.get(featureKey);;
        if(!featureControlStreamClient.initialized()){
            logger.warn("FeatureFlow is not initialized yet.");
        }
        if(control == null){
            logger.warn("Control does not exist, returning failover variant of " + failoverVariant);
            return failoverVariant;
        }

        //add featureflow.context
        addAdditionalContext(featureflowContext);
        
        String variant = control.evaluate(featureflowContext);
        return variant;

    }

    private void addAdditionalContext(FeatureflowContext featureflowContext) {
        featureflowContext.values.put(FeatureflowContext.FEATUREFLOW_HOUROFDAY, new JsonPrimitive(LocalTime.now().getHour()));
        featureflowContext.values.put(FeatureflowContext.FEATUREFLOW_DATE, new JsonPrimitive(FeatureflowContext.Builder.toIso(new DateTime())));
        
    }
    public void close() throws IOException {
    /*    this.eventProcessor.close();
        if (this.updateProcessor != null) {
            this.updateProcessor.close();
        }*/
    }

    public static Builder builder(String apiKey){
        return new Builder(apiKey);
    }

    public static class Builder {
        private FeatureflowConfig config = null;
        private String apiKey;
        private Map<CallbackEvent, List<FeatureControlCallbackHandler>> featureControlCallbackHandlers = new HashMap<>();
        private List<Feature> features = new ArrayList<>();



        public Builder (String apiKey){
            this.apiKey = apiKey;
        }

        public Builder withUpdateCallback(FeatureControlCallbackHandler featureControlCallbackHandler){
            this.withCallback(CallbackEvent.UPDATED_FEATURE, featureControlCallbackHandler);
            return this;
        }
        public Builder withDeleteCallback(FeatureControlCallbackHandler featureControlCallbackHandler){
            this.withCallback(CallbackEvent.DELETED_FEATURE, featureControlCallbackHandler);
            return this;
        }
        @Deprecated //use withUpdate or withDelete callbacks e.g. .withUpdateCallback(control -> System.out.println(control.getKey()))
        public Builder withCallback(FeatureControlCallbackHandler featureControlCallbackHandler){
            withUpdateCallback(featureControlCallbackHandler);
            return this;
        }
        public Builder withCallback(CallbackEvent event, FeatureControlCallbackHandler featureControlCallbackHandler){
            if(featureControlCallbackHandlers.get(event)==null){
                featureControlCallbackHandlers.put(event, new ArrayList<>());
            }
            this.featureControlCallbackHandlers.get(event).add(featureControlCallbackHandler);
            return this;
        }

        public Builder withConfig(FeatureflowConfig config){
            this.config = config;
            return this;
        }

        public Builder withFeature(Feature feature){
            this.features.add(feature);
            return this;
        }
        public Builder withFeatures(List<Feature> features){
            this.features = features;
            return this;
        }

        public FeatureflowClient build(){
            if(config==null){ config = new FeatureflowConfig.Builder().build();}
            return new FeatureflowClient(apiKey, features, config, featureControlCallbackHandlers);
        }
    }

    public class Evaluate {
        private String evaluateResult;
        private String featureKey;
        private FeatureflowContext context;

        Evaluate(FeatureflowClient featureflowClient, String featureKey, FeatureflowContext featureflowContext) {
            this.featureKey = featureKey;
            this.context = featureflowContext;
            evaluateResult = featureflowClient.eval(featureKey, featureflowContext);
        }
        public boolean isOn(){
            return is(Variant.on);
        }
        public boolean isOff(){
            return is(Variant.off);
        }
        public boolean is(String variant){
            eventHandler.queueEvent(new Event(featureKey, Event.EVALUATE_EVENT, context, evaluateResult, variant));
            return variant.equals(evaluateResult);
        }
        public String value(){
            eventHandler.queueEvent(new Event(featureKey, Event.EVALUATE_EVENT, context, evaluateResult, null));
            return evaluateResult;
        }
    }
}
