package io.eosnova.wallet.android.sdk

import android.app.Activity
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.net.Uri
import android.util.Log
import androidx.appcompat.app.AppCompatActivity
import com.google.gson.Gson
import com.google.gson.JsonObject
import com.google.gson.JsonParser
import io.eosnova.wallet.android.sdk.utils.AutoClearedBroadcast
import io.eosnova.wallet.android.sdk.utils.Utils
import java.util.*
import kotlin.collections.HashMap
import kotlin.collections.LinkedHashMap

/**
 * Created by shinhyo.
 */
abstract class BaseAuth(private val activity: Activity) {

    private val packageNameNova = "io.eosnova.wallet.android"
    private val action = "io.eosnova.wallet.android.sdk.receiver"

    abstract val chain: Chain
    var novaListener: OnNovaListener? = null
    private var isRegister: Boolean = false
    protected val gson by lazy { Gson() }

    init {
        if(!Utils.checkMinNovaWalletVersion(activity.applicationContext)) {
            val versionCode = Utils.getNovaWalletVersionCode(activity.applicationContext)
            val strVersion = if(versionCode < 0) "Unknown"  else versionCode.toString()
            val message =
                    "..\n" +
                    "************** NovaWallet SDK Warnning **************\n" +
                    "## This is not the minimum available version of NovaWallet.\n" +
                    "## If you want to check the minimum support version during development, call the \'Utils.checkMinNovaWalletVersion()\' function.\n" +
                    "- Require Minimum NovaWallet VersionCode : ${Utils.MIN_WALLET_VERSION}\n" +
                    "- The version code of the NovaWallet found : $strVersion"

            Log.w(Utils.TAG, message)
        }
    }

    private fun chkInit(): Boolean {
        if (!Utils.isInstalled(activity, packageNameNova)) {
            Utils.goStore(activity, packageNameNova)
            return true
        }

        chkListener()

        if (activity is AppCompatActivity) {
            val disposable = AutoClearedBroadcast(activity.lifecycle, activity, receiver)
            disposable.add(action)
            activity.lifecycle.addObserver(disposable)
        } else {
            if (!isRegister) {
                throw RuntimeException("Activity is required by register/unregister.")
            }
        }

        return false
    }

    fun register() {
        val filter = IntentFilter(action)
        filter.priority = 1000
        activity.registerReceiver(receiver, filter)
        isRegister = true
    }

    fun unregister() {
        try {
            activity.unregisterReceiver(receiver)
        } catch (e: Exception) {
            e.printStackTrace()
        }
        isRegister = false
    }

    fun requestTestMode(url: String) {
        send("test_url", gson.toJson(JsonObject().apply {
            addProperty("chain", chain.name)
            addProperty("url", url)
        }))
    }

    protected fun send(path: String, data: String = "") {
        send2Nova(makeUri(path).apply { if (data.isNotEmpty()) append("&data=").append(data) })
    }

    private fun send2Nova(sb: StringBuilder) {
        if (chkInit()) {
            return
        }

        Log.i("TEST", "uri $sb")
        activity.startActivity(Intent(Intent.ACTION_VIEW, Uri.parse(sb.toString())))
    }

    private fun makeUri(path: String): StringBuilder {
        return StringBuilder().apply {
            append("eosnova://auth")
            append("/").append(path)
            append("?packageName=").append(activity.applicationInfo.packageName)
            append("&protocol=").append("nova")
            append("&chain=").append(chain.name.toLowerCase(Locale.ENGLISH))
            append("&os=").append("android")
            append("&sdk_version=").append(BuildConfig.VERSION_NAME)
            append("&sdk_version_code=").append(BuildConfig.VERSION_CODE)
        }
    }


    private fun chkListener() {
        if (novaListener == null) throw RuntimeException("Listener is null")
    }

    private val receiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context?, intent: Intent) {
            val code = intent.getIntExtra("code", -1)
            val raw = intent.getStringExtra("raw") ?: ""

            val map = LinkedHashMap<String, String>()
            map["code"] = code.toString()

            when(code) {
                NovaSdkResult.ResultCode.SUCCESS -> {
                    map["msg"] = NovaSdkResult.SUCCESS.message
                    val jsonObject = JsonParser().parse(raw).asJsonObject
                    jsonObject.get("account_name")?.let { map["name"] = it.asString }
                    jsonObject.get("core_liquid_balance")?.let { map["balance"] = it.asString }
                    jsonObject.get("transaction_id")?.let { map["transaction_id"] = it.asString }
                }
                NovaSdkResult.ResultCode.CANCEL -> {
                    map["msg"] = NovaSdkResult.CANCEL.message
                }
                NovaSdkResult.ResultCode.REQUEST_PARAM_WRONG -> {
                    map["msg"] = NovaSdkResult.REQUEST_PARAM_WRONG.message
                }
                NovaSdkResult.ResultCode.EMPTY_ACCOUNT -> {
                    map["msg"] = NovaSdkResult.EMPTY_ACCOUNT.message
                }
                NovaSdkResult.ResultCode.NOT_FOUND_ACCOUNT_IN_NOVA -> {
                    map["msg"] = NovaSdkResult.NOT_FOUND_ACCOUNT_IN_NOVA.message
                }
                NovaSdkResult.ResultCode.NOT_FOUND_ACCOUNT_IN_CHAIN -> {
                    map["msg"] = NovaSdkResult.NOT_FOUND_ACCOUNT_IN_CHAIN.message
                }
                NovaSdkResult.ResultCode.UNKNOWN_CHAIN_TYPE -> {
                    map["msg"] = NovaSdkResult.UNKNOWN_CHAIN_TYPE.message
                }
                NovaSdkResult.ResultCode.FAIL_TRANSACTION -> {
                    map["msg"] = NovaSdkResult.FAIL_TRANSACTION.message
                }
                NovaSdkResult.ResultCode.FAIL_SIGNATURE -> {
                    map["msg"] = NovaSdkResult.FAIL_TRANSACTION.message
                }
                NovaSdkResult.ResultCode.USER_BACK_DAPP -> {
                    map["msg"] = NovaSdkResult.USER_BACK_DAPP.message
                }
                NovaSdkResult.ResultCode.NOT_USE -> {
                    map["msg"] = NovaSdkResult.NOT_USE.message
                }
            }

            if(raw.isNotEmpty()) {
                val msg = map["msg"] ?: ""
                if(msg != raw) {
                    map["raw"] = raw
                }
            }

            /*
            raw?.let {
                map["raw"] = it
            }
             */
            novaListener?.callback(map)
        }

    }

}
