package io.emergentlabs.emergent;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.SystemClock;
import android.provider.Settings.Secure;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import org.bson.types.ObjectId;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import java.io.File;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.RejectedExecutionException;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Response;
import okhttp3.ResponseBody;

import java.io.IOException;

/**
 * Session data about the running Android app which includes
 * information that doesn't change over time (app name, app version, manufacturer, etc.) and custom user params.
 *
 */
public class Session implements Callback {

    private static final Long startTime = SystemClock.elapsedRealtime();
    private static final String GEOLOOKUP = "https://cryptic-taiga-14181.herokuapp.com/lookup/";
    private static final int MAX_RESEND_ATTEMPTS = 5;

    @Nullable
    private String appName;
    @Nullable
    private String appVersion;
    @Nullable
    private String manufacturer;
    @Nullable
    private String model;
    @Nullable
    private String codeName;
    @Nullable
    private String deviceName;
    @Nullable
    private String deviceId;
    @Nullable
    private String packageName;

    private String platform = "android";

    private String versionRelease = Build.VERSION.RELEASE;

    private ConnectionInfo connectionInfo;

    private Long uptime;
    private String key;
    private String id = new ObjectId().toString();

    private transient Integer resendAttempts = 0;
    private transient Preferences prefs;
    private transient ApiClient apiClient;
    private transient Context context;

    @NonNull
    private final Map<String, Boolean> customBooleans;
    @NonNull
    private final Map<String, Integer> customInts;
    @NonNull
    private final Map<String, String> customStrings;

    public Session(final ApiClient apiClient, final Context context,
            final String apiKey,
            final boolean sendAfter) {

        this.prefs = new Preferences(context);
        this.apiClient = apiClient;
        this.context = context;
        key = apiKey;
        deviceId = Secure.getString(context.getContentResolver(), Secure.ANDROID_ID);
        setAppName(context);
        setPackageName(context);
        appVersion = getAppVersion(context);

        connectionInfo = Connectivity.getConnectionInfo(context);

        customBooleans = new ConcurrentHashMap<String, Boolean>();
        customInts = new ConcurrentHashMap<String, Integer>();
        customStrings = new ConcurrentHashMap<String, String>();

        DeviceName.with(context).request(new DeviceName.Callback() {
            @Override
            public void onFinished(DeviceName.DeviceInfo info, Exception error) {
                setManufacturer(info.manufacturer);
                setModel(info.model);
                setCodeName(info.codename);
                setDeviceName(info.getName());
                if (sendAfter) {
                    flush();
                }
            }
        });
    }

    private JsonObject storeGeoInfo(final JsonObject json, final JsonObject result) {
        Logger.debug("Got JSON response: " + result);
        final JsonObject cityObj = result.get("City").getAsJsonObject();
        if (cityObj == null) {
            Logger.error("Could not fetch geo info: invalid JSON; not formatted properly");
            return json;
        }
        final JsonObject names = cityObj.get("Names").getAsJsonObject();
        final String city = names.get("en").getAsString();
        final JsonObject countryObj = result.get("Country").getAsJsonObject();
        final String country = countryObj.get("IsoCode").getAsString();
        Logger.debug("Got geo info; city is " + city + " country is " + country);
        json.addProperty("city", city);
        json.addProperty("country", country);
        Logger.debug("Session json is " + json);
        return json;
    }

    private Callback geoCallback(final ApiClient apiClient, final Context context) {

        final Session session = this;
        final String sessionUri = apiClient.createUri("session/new");
        final JsonObject json = convertToJson();

        return new Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
                //do something to indicate error
            }
            @Override
            public void onResponse(Call call, Response response) throws IOException {
                if (response.isSuccessful()) {
                    final String responseData = response.body().string();
                    final JsonObject result = (new JsonParser()).parse(responseData).getAsJsonObject();
                    if (result == null || result.get("Country") == null) {
                        // for whatever reason, we were unable to get
                        // geo info; send the session request anyway
                        apiClient.sendRequest(context, sessionUri,
                                json, session);
                        return;
                    }
                    apiClient.sendRequest(context, sessionUri, storeGeoInfo(json, result), session);
                }
            }
        };
    }

    @Override
    public void onFailure(Call call, IOException e) {
        //do something to indicate error
    }

    @Override
    public void onResponse(Call call, Response response) throws IOException {
        if (!response.isSuccessful()) {
            if (resendAttempts >= MAX_RESEND_ATTEMPTS) {
                Logger.error("Couldn't send session data after " +
                        resendAttempts + " attempts");
                return;
            }
            this.resendAttempts++;
            flush();
            return;
        }
        try (ResponseBody responseBody = response.body()) {
            if (!response.isSuccessful()) throw new IOException("Unexpected code " + response);
            final String responseData = responseBody.string();
            Logger.debug("RESPONSE: " + responseData);
            final JsonObject result = (new JsonParser()).parse(responseData).getAsJsonObject();
            if (result == null || result.get("error") != null) {
                if (result.get("error") != null) {
                    Logger.error("ERROR: " + result.get("error"));
                    return;
                }
            }
            final String appId = result.get("appId").getAsString();
            final String sessionId = result.get("id").getAsString();
            final String token = result.get("token").getAsString();
            Logger.debug(String.format("App Id %s Session id %s token %s",
                        appId, sessionId, token));
            prefs.setAppId(appId);
            prefs.setSessionId(sessionId);
            prefs.setSessionToken(token);
        }
    }

    public JsonObject convertToJson() {
        final Gson gson = new Gson();
        final String json = gson.toJson(this);
        final JsonObject jsonObject = (new JsonParser()).parse(json).getAsJsonObject();
        jsonObject.add("connectionInfo", new Gson().toJsonTree(getConnectionInfo()));
        Logger.debug("Session json is " + json);
        return jsonObject;
    }

    private void flush() {
        final Session session = this;
        final JsonObject json = convertToJson();
        try {
            Async.run(new Runnable() {
                @Override
                public void run() {
                    final Drawable appIcon = Utils.getAppIcon(context);
                    final Bitmap bm = Utils.drawableToBitmap(appIcon);
                    final File iconFile = Utils.saveBitmapToFile(context, "appicon", bm);
                    Logger.debug("Saved app icon file to " + iconFile.toString());
                    apiClient.sendRequest(context, GEOLOOKUP, json,
                            geoCallback(apiClient, context));
                }
            });
        } catch (RejectedExecutionException e) {
            Logger.warn("Failed to flush all on-disk errors, retaining unsent errors for later.");
        }
    }

    public void setCustomInteger(final String key, final Integer value) {
        customInts.put(key, value);
    }

    public void setCustomBoolean(final String key, final Boolean value) {
        customBooleans.put(key, value);
    }

    public void setCustomString(final String key, final String value) {
        customStrings.put(key, value);
    }

    public void setManufacturer(final String manufacturer) {
        this.manufacturer = manufacturer;
    }

    public void setKey(final String key) {
        this.key = key;
    }

    public void setModel(final String model) {
        this.model = model;
    }

    public void setCodeName(final String codeName) {
        this.codeName = codeName;
    }

    public void setDeviceName(final String deviceName) {
        this.deviceName = deviceName;
    }

    public void setUptime(final Long uptime) {
        this.uptime = uptime;
    }

    public void setDeviceId(final String deviceId) {
        this.deviceId = deviceId;
    }

    public void setPlatform(final String platform) {
        this.platform = platform;
    }

    public void setVersionRelease(final String versionRelease) {
        this.versionRelease = versionRelease;
    }

    public void setConnectionInfo(final ConnectionInfo connectionInfo) {
        this.connectionInfo = connectionInfo;
    }

    private void setAppName(final Context context) {
        try {
            final ApplicationInfo applicationInfo = context.getApplicationInfo();
            int stringId = applicationInfo.labelRes;
            this.appName = stringId == 0 ? applicationInfo.nonLocalizedLabel.toString() : context.getString(stringId);
        } catch (Exception e) {
            Logger.error("Could not determine app name", e);
        }
    }

    private void setPackageName(final Context context) {
        try {
            this.packageName = context.getPackageName();
        } catch (Exception e) {
            Logger.error("Could not determine package name", e);
        }
    }

    private String getAppVersion(final Context context) {
        try {
            final PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            return pInfo.versionName;
        } catch (PackageManager.NameNotFoundException nne) {
            Logger.error("Could not find package", nne);
        }
        return "unknown";
    }

    public String getAppName() {
        return appName;
    }

    public String getPackageName() {
        return packageName;
    }

    public String getDeviceName() {
        return deviceName;
    }

    public String getDeviceId() {
        return deviceId;
    }

    public String getModel() {
        return model;
    }

    public String getManufacturer() {
        return manufacturer;
    }

    public String getPlatform() {
        return platform;
    }

    public String getVersionRelease() {
        return versionRelease;
    }

    public ConnectionInfo getConnectionInfo() {
        return connectionInfo;
    }

    public String getCodeName() {
        return codeName;
    }

    public String getKey() {
        return key;
    }

    public Long getUptime() {
        return SystemClock.elapsedRealtime() - startTime;
    }

    public String osName() {
        return Build.VERSION_CODES.class.getFields()[android.os.Build.VERSION.SDK_INT].getName();
    }
}
