package io.emergentlabs.emergent;

import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.widget.Toast;


import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import java.io.InputStream;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;

import java.io.IOException;

/**
 * Emergent configuration, contains client-level information
 *
 */
public class Config implements Parcelable, Callback {
    @NonNull
    private final String apiKey;

    // 2 minutes
    private static final int globalConfigPollInterval = 30000;
    private static final boolean USE_EXECUTOR = true;

    private final OkHttpClient httpClient = new OkHttpClient();
    private ConfigCallback configCallback;

    private String  apiAddr;
    private Boolean enableLogging;
    private Boolean enableNetworkStats;
    private Boolean enableExceptionReporting = true;

    private boolean useExecutor;

    @NonNull
    private HandlerThread handlerThread;

    @Nullable
    private Handler handler;

    @NonNull
    private Runnable runnable;

    private ScheduledThreadPoolExecutor executor;
    private ScheduledFuture schedule;

    private static final String S3_ADDR = "https://configemergent.s3.amazonaws.com/%s.cloud.yaml.gz";

    public Config(@NonNull String apiKey, ConfigCallback configCallback) {
        this(apiKey, true, configCallback);
    }

    public Config(@NonNull String apiKey, final boolean useExecutor, ConfigCallback configCallback) {
        this.apiKey = apiKey;
        this.configCallback = configCallback;
        this.useExecutor = useExecutor;

        if (useExecutor) {

        } else {
            handlerThread = new HandlerThread("HandlerThread");
            handlerThread.start();
            handler = new Handler(handlerThread.getLooper());
        }
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(apiAddr);
        out.writeString(apiKey);
        out.writeByte((byte) (enableLogging ? 1 : 0));
        out.writeByte((byte)(enableNetworkStats ? 1 : 0));
        out.writeByte((byte)(enableExceptionReporting ? 1 : 0));
    }

    public Config setParams(final ConfigParams params) {
        if (params == null) {
            Logger.error("Could not update config; null params");
            return null;
        }

        this.apiAddr = params.getApiAddr();
        this.enableNetworkStats = params.getEnableNetworkStats();
        this.enableLogging = params.getEnableLogging();
        this.enableExceptionReporting = params.getEnableExceptionReporting();

        Logger.debug(String.format("updating config: enable logging %b enable network stats %b", enableLogging, enableNetworkStats));
        return this;
    }

    private Config(Parcel in) {
        apiAddr = in.readString();
        apiKey = in.readString();
        enableLogging = in.readByte() != 0;
        enableNetworkStats = in.readByte() != 0;
        enableExceptionReporting = in.readByte() != 0;
    }

    public static final Parcelable.Creator<Config> CREATOR = new Parcelable.Creator<Config>() {
        public Config createFromParcel(Parcel in) {
            return new Config(in);
        }

        public Config[] newArray(int size) {
            return new Config[size];
        }
    };

    public void poll(final Context context) {
        runnable = new Runnable() {
            public void run() {
                LogUtils.extractLog(context);
                Logger.debug("Current log file size: " + LogUtils.logFileSize(context));
                fetchLatest(context);
                handler.postDelayed(runnable, globalConfigPollInterval);
            }
        };
        if (useExecutor) {
            schedule = executor.scheduleWithFixedDelay(runnable, 0, 30, TimeUnit.SECONDS);
        } else {
            handler.postDelayed(runnable, globalConfigPollInterval);
        }
    }

    public void stopPolling() {
        if (handler != null) {
            handler.removeCallbacks(runnable);
        }
        if (schedule != null) {
            // If cancel is false, the current iteration will finish first
            schedule.cancel(false);
        }
        if (executor != null) {
            // shutdown() allows the final iteration to finish
            executor.shutdown();
        }
    }

    @Override
    public void onFailure(Call call, IOException e) {
        Logger.error("Error fetching latest config " + e.getMessage());
    }

    @Override
    public void onResponse(Call call, Response response) throws IOException {
        if (response.isSuccessful()) {
            final InputStream is = response.body().byteStream();
            if (is == null) {
                Logger.error("Unable to process config response body");
                return;
            }
            final Config config = setParams(parseConfig(is));
            if (config != null) {
                configCallback.onNewConfig(config);
            }
        }
    }

    public void fetchLatest(final Context context) {

        final String configAddr = configAddr();
        Logger.debug("Fetching latest config; url is " + configAddr);

        final Request request = new Request.Builder()
            .url(configAddr)
            .build();

        httpClient.newCall(request).enqueue(this);
    }

    private ConfigParams parseConfig(final InputStream yamlResult) {
        //final String response = yamlResult;
        final String response = Utils.decompress(yamlResult);
        if (response == null) {
            return null;
        }
        Logger.debug("Response from server for config fetch:" + response);
        try {
            final ObjectMapper mapper = new ObjectMapper(new YAMLFactory());
            final ConfigParams params = (ConfigParams)mapper.readValue(response, ConfigParams.class);

            Logger.debug("Loaded Emergent config: " + params);
            return params;
        } catch (Exception e) {
            Logger.error("Could not load emergent config", e);
        }

        return null;
    }

    @NonNull
    public String getApiAddr() {
        return apiAddr;
    }

    @NonNull
    public String configAddr() {
        return String.format(S3_ADDR, getApiKey());
    }

    @NonNull
    public String getApiKey() {
        return apiKey;
    }

    /*
     * Whether or not to submit logs to Emergent server
     *
     * @return enableLogging
     */
    public Boolean enableLogging() {
        return enableLogging;
    }

    /*
     * Whether or not to report network stats to Emergent server
     *
     * @return enableNetworkStats
     */
    public Boolean enableNetworkStats() {
        return enableNetworkStats;
    }

    /*
     * Whether or not to report crashes to Emergent server
     *
     * @return enableExceptionReporting
     */
    public Boolean enableExceptionReporting() {
        return enableExceptionReporting;
    }
}
