package io.emergentlabs.emergent;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import io.emergentlabs.emergent.s3.Constants;
import io.emergentlabs.emergent.s3.Util;

import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.mobileconnectors.s3.transferutility.TransferListener;
import com.amazonaws.mobileconnectors.s3.transferutility.TransferObserver;
import com.amazonaws.mobileconnectors.s3.transferutility.TransferState;
import com.amazonaws.mobileconnectors.s3.transferutility.TransferUtility;


/**
 * Emergent client, handles reporting session data and uploading logs
 *
 */
public class Client implements ConfigCallback {

    private static final String NAMESPACE = "io.emergentlabs.emergent";
    private static final String API_KEY   = NAMESPACE + ".API_KEY";

    private Context context;
    private ApiClient apiClient;
    private final Config config;
    private final String apiKey;
    private final Preferences prefs;
    private Session session;

    // The TransferUtility is the primary class for managing transfer to S3
    private TransferUtility transferUtility;

    /**
     * Initialize Emergent client
     *
     * @param context an Android context, usually <code>this</code>
     */
    public Client(@NonNull Context context) {
        this(context, true);
    }

    /**
     * Initialize a Emergent client
     *
     * @param context         an Android context, usually <code>this</code>
     * @param enableExceptionHandler should we automatically handle uncaught exceptions?
     */
    public Client(@NonNull Context c, boolean enableExceptionHandler) {

        context = c.getApplicationContext();

        transferUtility = Util.getTransferUtility(context);

        prefs = new Preferences(context);
        apiKey = loadApiKey(context);
        config = new Config(apiKey, this);
    }

    public void pollConfig(final Context context) {
        config.poll(context);
    }

    public void stop() {
        Logger.debug("Received notification to stop client");
        if (config != null) {
            config.stopPolling();
        }
    }

    private static String loadApiKey(final Context context) {
        try {
            final ApplicationInfo ai = context.getPackageManager()
                .getApplicationInfo(context.getPackageName(),
                        PackageManager.GET_META_DATA);

            final Bundle data = ai.metaData;
            return data.getString(API_KEY);
        } catch (Exception e) {
            throw new NullPointerException("You must provide an Emergent API key");
        }
    }

    @Override
    public void onNewConfig(final Config c) {
        if (c == null) {
            Logger.error("New config could not be processed");
            return;
        }
        if (apiClient == null) {
            apiClient = new ApiClient(context, config.getApiAddr());
            session = new Session(apiClient, context, apiKey, true);
        }
        if (c.enableLogging()) {
            sendLog();
        } else {
            Logger.debug("Logging disabled");
        }
    }

    public void setCustomInteger(final String key, final Integer value) {
        if (session != null) {
            session.setCustomInteger(key, value);
        }
    }

    public void setCustomString(final String key, final String value) {
        if (session != null) {
            session.setCustomString(key, value);
        }
    }

    public void setCustomBoolean(final String key, final Boolean value) {
        if (session != null) {
            session.setCustomBoolean(key, value);
        }
    }

    public void updateConfig(final Config config) {
        if (config.enableLogging()) {
            final String sessionId = prefs.getSessionId();
            if (TextUtils.isEmpty(sessionId)) {
                Logger.debug("No session id found");
                return;
            }
        }
    }

    /**
     * Send log POSTs the current application log
     * to the Emergent analytics server
     *
     */
    public void sendLog() {
        try {
            final String token = prefs.getSessionToken();
            final String logUri = apiClient.createUri("session/log/upload");
            if (TextUtils.isEmpty(token)) {
                Logger.debug("No session token found, not sending log file yet");
                return;
            }
            final File file = LogUtils.getLogFile(context);
            final String appId = prefs.getAppId();
            final String sessionId = prefs.getSessionId();
            final String time = String.valueOf(System.currentTimeMillis());
            final String objectKey = String.format("%s/%s-%s.log", appId, sessionId, time);

            final ObjectMetadata metadata = new ObjectMetadata();
            final Map<String, String> logMetadata = new HashMap<String, String>();
            logMetadata.put("sessionId", sessionId);
            logMetadata.put("time", time);
            metadata.setUserMetadata(logMetadata);
            TransferObserver observer = transferUtility.upload(Constants.BUCKET_NAME, objectKey, file, metadata);
            observer.setTransferListener(new UploadListener());
        } catch (Exception e) {
            Logger.error("Unable to send log file", e);
        }
    }

        /*
     * A TransferListener class that can listen to a upload task and be notified
     * when the status changes.
     */
    private class UploadListener implements TransferListener {

        // Simply updates the UI list when notified.
        @Override
        public void onError(int id, Exception e) {
            Logger.error("Error during upload: " + id, e);
        }

        @Override
        public void onProgressChanged(int id, long bytesCurrent, long bytesTotal) {
            Logger.debug(String.format("onProgressChanged: %d, total: %d, current: %d",
                    id, bytesTotal, bytesCurrent));
        }

        @Override
        public void onStateChanged(int id, TransferState newState) {
            Logger.debug("onStateChanged: " + id + ", " + newState);
            if (newState == TransferState.COMPLETED) {
                Logger.debug("Clearing log file");
                LogUtils.clearLogfile(context);
            }
        }
    }
}
