package io.emergentlabs.emergent;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.telephony.TelephonyManager;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * Check device's network connectivity and speed
 *
 */
public class Connectivity {

    /**
     * Get the network info
     * @param context
     * @return
     */
    public static NetworkInfo getNetworkInfo(Context context){
        ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        return cm.getActiveNetworkInfo();
    }

    /**
     * Check if there is any connectivity
     * @param context
     * @return
     */
    public static boolean isConnected(Context context){
        NetworkInfo info = Connectivity.getNetworkInfo(context);
        return (info != null && info.isConnected());
    }

    /**
     * Check if there is any connectivity to a Wifi network
     * @param context
     * @param type
     * @return
     */
    public static boolean isConnectedWifi(Context context){
        NetworkInfo info = Connectivity.getNetworkInfo(context);
        return (info != null && info.isConnected() && info.getType() == ConnectivityManager.TYPE_WIFI);
    }

    public static String getIPAddress(boolean useIPv4) {
        try {
            List<NetworkInterface> interfaces = Collections.list(NetworkInterface.getNetworkInterfaces());
            for (NetworkInterface intf : interfaces) {
                List<InetAddress> addrs = Collections.list(intf.getInetAddresses());
                for (InetAddress addr : addrs) {
                    if (!addr.isLoopbackAddress()) {
                        String sAddr = addr.getHostAddress();
                        //boolean isIPv4 = InetAddressUtils.isIPv4Address(sAddr);
                        boolean isIPv4 = sAddr.indexOf(':')<0;

                        if (useIPv4) {
                            if (isIPv4)
                                return sAddr;
                        } else {
                            if (!isIPv4) {
                                int delim = sAddr.indexOf('%'); // drop ip6 zone suffix
                                return delim<0 ? sAddr.toUpperCase() : sAddr.substring(0, delim).toUpperCase();
                            }
                        }
                    }
                }
            }
        } catch (Exception ex) { } // for now eat exceptions
        return "";
    }

    /**
     * Check if there is any connectivity to a mobile network
     * @param context
     * @param type
     * @return
     */
    public static boolean isConnectedMobile(Context context){
        NetworkInfo info = Connectivity.getNetworkInfo(context);
        return (info != null && info.isConnected() && info.getType() == ConnectivityManager.TYPE_MOBILE);
    }

    /**
     * Check if there is fast connectivity
     * @param context
     * @return
     */
    public static ConnectionInfo getConnectionInfo(Context context){
        final ConnectionInfo connectionInfo = new ConnectionInfo();
        final NetworkInfo networkInfo = Connectivity.getNetworkInfo(context);
        if (networkInfo != null && networkInfo.isConnected()) {
            connectionInfo.setIsConnected(true);
            final String networkType =
                getNetworkType(networkInfo.getType(), networkInfo.getSubtype());
            final String networkSpeed =
                getNetworkSpeed(networkInfo.getType(), networkInfo.getSubtype());
            final String ipAddress = getIPAddress(true); // IPv4
            connectionInfo.setIpAddress(ipAddress);
            connectionInfo.setNetworkType(networkType);
            connectionInfo.setNetworkSpeed(networkSpeed);
        } else {
            connectionInfo.setIsConnected(false);
        }
        return connectionInfo;
    }

    public static String getNetworkType(int type, int subType) {
        if (type == ConnectivityManager.TYPE_WIFI) {
            return "wifi";
        } else if (type == ConnectivityManager.TYPE_MOBILE) {
            return "mobile";
        } else {
            return "unknown";
        }
    }

    /**
     * Check if the connection is fast
     * @param type
     * @param subType
     * @return
     */
    public static String getNetworkSpeed(int type, int subType){
        if (type == ConnectivityManager.TYPE_WIFI) {
            return "";
        } else if (type==ConnectivityManager.TYPE_MOBILE) {
            switch (subType) {
                case TelephonyManager.NETWORK_TYPE_1xRTT:
                    return "50-100kbps";
                case TelephonyManager.NETWORK_TYPE_CDMA:
                    return "14-64kbps";
                case TelephonyManager.NETWORK_TYPE_EDGE:
                    return "50-100kbps";
                case TelephonyManager.NETWORK_TYPE_EVDO_0:
                    return "400-1000kbps";
                case TelephonyManager.NETWORK_TYPE_EVDO_A:
                    return "600-1400kbps";
                case TelephonyManager.NETWORK_TYPE_GPRS:
                    return "100kbps";
                case TelephonyManager.NETWORK_TYPE_HSDPA:
                    return "2-14Mbps";
                case TelephonyManager.NETWORK_TYPE_HSPA:
                    return "700-1700kbps";
                case TelephonyManager.NETWORK_TYPE_HSUPA:
                    return "1-23Mbps";
                case TelephonyManager.NETWORK_TYPE_UMTS:
                    return "400-7000kbps";
                    /*
                     * Above API level 7, make sure to set android:targetSdkVersion
                     * to appropriate level to use these
                     */
                case TelephonyManager.NETWORK_TYPE_EHRPD: // API level 11
                    return "1-2 Mbps";
                case TelephonyManager.NETWORK_TYPE_EVDO_B: // API level 9
                    return "5 Mbps";
                case TelephonyManager.NETWORK_TYPE_HSPAP: // API level 13
                    return "10-20 Mbps";
                case TelephonyManager.NETWORK_TYPE_IDEN: // API level 8
                    return "~25 kbps";
                case TelephonyManager.NETWORK_TYPE_LTE: // API level 11
                    return "10+ Mbps";
                case TelephonyManager.NETWORK_TYPE_UNKNOWN:
                default:
                    return "";
            }
        } else {
            return "";
        }
    }

}
