package io.dinject.controller;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.UUID;

/**
 * Helper type conversion methods.
 * <p/>
 * These methods are intended to be used by APT source generators.
 */
public class PathTypeConversion {

  /**
   * Convert to int.
   */
  public static int asInt(String value) {
    return Integer.valueOf(value);
  }

  /**
   * Convert to long.
   */
  public static long asLong(String value) {
    return Long.valueOf(value);
  }

  /**
   * Convert to double.
   */
  public static double asDouble(String value) {
    return Double.valueOf(value);
  }

  /**
   * Convert to float.
   */
  public static float asFloat(String value) {
    return Float.valueOf(value);
  }

  /**
   * Convert to boolean.
   */
  public static boolean asBool(String value) {
    return Boolean.valueOf(value);
  }

  /**
   * Convert to BigDecimal (not nullable).
   */
  public static BigDecimal asDecimal(String value) {
    return new BigDecimal(value);
  }

  /**
   * Convert to LocalDate (not nullable).
   */
  public static LocalDate asDate(String value) {
    return LocalDate.parse(value);
  }

  /**
   * Convert to LocalTime (not nullable).
   */
  public static LocalTime asTime(String value) {
    return LocalTime.parse(value);
  }

  /**
   * Convert to LocalDateTime (not nullable).
   */
  public static LocalDateTime asDateTime(String value) {
    return LocalDateTime.parse(value);
  }

  /**
   * Convert to UUID (not nullable).
   */
  public static UUID asUUID(String value) {
    return UUID.fromString(value);
  }

  /**
   * Convert to Integer (not nullable).
   */
  public static Integer asInteger(String value) {
    return Integer.valueOf(value);
  }

  /**
   * Convert to Integer (allowing nulls).
   */
  public static Integer toInteger(String value) {
    if (isNullOrEmpty(value)) {
      return null;
    }
    return Integer.valueOf(value);
  }

  /**
   * Convert to Long (allowing nulls).
   */
  public static Long toLong(String value) {
    if (isNullOrEmpty(value)) {
      return null;
    }
    return Long.valueOf(value);
  }

  /**
   * Convert to Boolean (allowing nulls).
   */
  public static Boolean toBoolean(String value) {
    if (isNullOrEmpty(value)) {
      return null;
    }
    return Boolean.valueOf(value);
  }

  /**
   * Convert to UUID (allowing nulls).
   */
  public static UUID toUUID(String value) {
    if (isNullOrEmpty(value)) {
      return null;
    }
    return UUID.fromString(value);
  }

  /**
   * Convert to LocalDate (allowing nulls).
   */
  public static LocalDate toLocalDate(String value) {
    if (isNullOrEmpty(value)) {
      return null;
    }
    return LocalDate.parse(value);
  }

  /**
   * Convert to LocalTime (allowing nulls).
   */
  public static LocalTime toTime(String value) {
    if (isNullOrEmpty(value)) {
      return null;
    }
    return LocalTime.parse(value);
  }

  /**
   * Convert to LocalDateTime (allowing nulls).
   */
  public static LocalDateTime toDateTime(String value) {
    if (isNullOrEmpty(value)) {
      return null;
    }
    return LocalDateTime.parse(value);
  }

  private static boolean isNullOrEmpty(String value) {
    return value == null || value.isEmpty();
  }

}
