package io.digitalpatterns.camunda.encryption;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.camunda.spin.Spin;
import org.camunda.spin.impl.json.jackson.format.JacksonJsonDataFormat;

import javax.crypto.Cipher;
import javax.crypto.SealedObject;
import java.io.Serializable;

import static io.digitalpatterns.camunda.encryption.EncryptionConstants.ALGORITHM_NAME;

/**
 * Class that encrypts a process variable
 */
@Slf4j
public class DefaultProcessInstanceSpinVariableEncryptor extends ProcessInstanceSpinVariable
        implements ProcessInstanceSpinVariableEncryptor {

    private JacksonJsonDataFormat formatter;


    public DefaultProcessInstanceSpinVariableEncryptor(JacksonJsonDataFormat formatter,
                                                       String passPhrase, String salt) {
        this.formatter = formatter;

        if (passPhrase == null) {
            throw new IllegalArgumentException("PassPhrase cannot be null");
        }
        if (salt == null) {
            throw new IllegalArgumentException("Salt cannot be null");
        }
        initialiseKeys(passPhrase, salt);

    }

    public DefaultProcessInstanceSpinVariableEncryptor(String passPhrase, String salt) {
        this(new JacksonJsonDataFormat("application/json", new ObjectMapper()), passPhrase, salt);
    }

    /**
     * Encrypts an object. It creates a Spin object and is encrypted
     *
     * @param object Object that needs to be encrypted
     * @return SealedObject
     * @see SealedObject
     */
    public SealedObject encrypt(Object object) {
        Spin<?> spinObject = Spin.S(object, formatter);
        return performSeal(spinObject.toString());
    }

    private SealedObject performSeal(Serializable object) {
        try {
            Cipher cipher = Cipher.getInstance(ALGORITHM_NAME);
            cipher.init(Cipher.ENCRYPT_MODE, this.secretKey, this.pbeParamSpec);
            return new SealedObject(object, cipher);
        } catch (Exception e) {
            log.error("Failed to create sealed object", e);
            throw new EncryptionException(e);
        }
    }

}
