package io.digitalpatterns.camunda.encryption;

import lombok.extern.slf4j.Slf4j;
import org.camunda.bpm.engine.impl.javax.el.FunctionMapper;
import org.camunda.bpm.engine.impl.util.ReflectUtil;
import org.camunda.spin.Spin;

import javax.crypto.SealedObject;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

@Slf4j
public class SpinEncryptionFunctionMapper extends FunctionMapper {

    static ProcessInstanceSpinVariableDecryptor processInstanceSpinVariableDecryptor;
    static ProcessInstanceSpinVariableEncryptor processInstanceSpinVariableEncryptor;

    private static Map<String, Method> encryptionFunctionMap = null;

    private void ensureContextFunctionMapInitialized() {
        if (encryptionFunctionMap == null) {
            synchronized (SpinEncryptionFunctionMapper.class) {
                if (encryptionFunctionMap == null) {
                    encryptionFunctionMap = new HashMap<>();
                    createMethodBindings();
                }
            }
        }
    }

    private void createMethodBindings() {
        Class<?> mapperClass = getClass();
        encryptionFunctionMap.put("encrypt",
                ReflectUtil.getMethod(mapperClass, "encrypt", Object.class));
        encryptionFunctionMap.put("decrypt",
                ReflectUtil.getMethod(mapperClass, "decrypt", Object.class));
    }

    @Override
    public Method resolveFunction(String prefix, String localName) {
        ensureContextFunctionMapInitialized();
        return encryptionFunctionMap.get(localName);
    }


    public static SealedObject encrypt(Object object) {
        return processInstanceSpinVariableEncryptor.encrypt(object);
    }

    public static Spin decrypt(Object sealedObject) {
        return processInstanceSpinVariableDecryptor.decrypt((SealedObject) sealedObject);
    }
}
