package io.digitalpatterns.camunda.encryption;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.camunda.spin.Spin;
import org.camunda.spin.impl.json.jackson.format.JacksonJsonDataFormat;

import javax.crypto.Cipher;
import javax.crypto.SealedObject;
import java.io.Serializable;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.spec.X509EncodedKeySpec;

import static io.digitalpatterns.camunda.encryption.EncryptionConstants.*;

/**
 * Class that encrypts a process variable
 */
@Slf4j
public class ProcessInstanceSpinVariableEncryptor {

    private JacksonJsonDataFormat formatter;
    private PublicKey publicKey;

    public ProcessInstanceSpinVariableEncryptor(JacksonJsonDataFormat formatter,
                                                String pathToPublicKey) {
        this.formatter = formatter;

        if (pathToPublicKey == null) {
            throw new IllegalArgumentException("Path to public key cannot be null");
        }
        this.publicKey = loadPublicKey(pathToPublicKey);
    }

    public ProcessInstanceSpinVariableEncryptor(String pathToPublicKey) {
        this(new JacksonJsonDataFormat("application/json", new ObjectMapper()), pathToPublicKey);
    }

    /**
     * Encrypts an object. It creates a Spin object and is encrypted
     * @param object Object that needs to be encrypted
     * @return SealedObject
     * @see SealedObject
     */
    public SealedObject encrypt(Object object) {
        Spin<?> spinObject = Spin.S(object, formatter);
        return performSeal(spinObject.toString());
    }

    private SealedObject performSeal(Serializable object) {
        try {
            Cipher instance = Cipher.getInstance(ALGORITHM_NAME + "/" + MODE_OF_OPERATION + "/" + PADDING_SCHEME);
            instance.init(Cipher.ENCRYPT_MODE, this.publicKey);
            return new SealedObject(object, instance);
        } catch (Exception e) {
            log.error("Failed to create sealed object", e);
            throw new EncryptionException(e);
        }
    }

    private PublicKey loadPublicKey(String pathToPublicKey) {
        try {
            byte[] keyAsBytes = Files.readAllBytes(Paths.get(pathToPublicKey));
            X509EncodedKeySpec spec = new X509EncodedKeySpec(keyAsBytes);
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            return keyFactory.generatePublic(spec);
        } catch (Exception e) {
            throw new EncryptionException(e);
        }
    }

}
