package io.digitalpatterns.camunda.encryption;

import lombok.extern.slf4j.Slf4j;
import org.camunda.bpm.engine.variable.VariableMap;
import org.camunda.bpm.engine.variable.impl.VariableMapImpl;
import org.camunda.spin.Spin;

import javax.crypto.SealedObject;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Map;

@Slf4j
public class ProcessInstanceSpinVariableDecryptor {

    private PrivateKey privateKey;

    public ProcessInstanceSpinVariableDecryptor(String pathToPrivateKey) {
        if (pathToPrivateKey == null) {
            throw new IllegalArgumentException("Path to private key cannot be null");
        }
        this.privateKey = loadPrivateKey(pathToPrivateKey);
    }


    /**
     *
     * @param sealedObject
     * @return
     */
    public Spin decrypt(SealedObject sealedObject) {
        try {
            log.info("Preparing to decrypt object");
            Object object = sealedObject.getObject(this.privateKey);
            return Spin.JSON(object);
        } catch (Exception e) {
            throw new DecryptionException(e);
        }
    }

    public VariableMap decrypt(Map<String, Object> variableMap) {
        variableMap.keySet().forEach(key -> {
            if (variableMap.get(key) instanceof SealedObject) {
                SealedObject variable = (SealedObject) variableMap.get(key);
                variableMap.replace(key, decrypt(variable));
            }
        });
        return new VariableMapImpl(variableMap);
    }

    private PrivateKey loadPrivateKey(String pathToPrivateKey) {
        try {
            byte[] keyAsBytes = Files.readAllBytes(Paths.get(pathToPrivateKey));
            PKCS8EncodedKeySpec spec = new PKCS8EncodedKeySpec(keyAsBytes);
            KeyFactory keyFactory = KeyFactory.getInstance("RSA");
            return keyFactory.generatePrivate(spec);
        } catch (Exception e) {
            throw new DecryptionException(e);
        }

    }
}
