package io.digitalpatterns.camunda.encryption;

import lombok.extern.slf4j.Slf4j;
import org.camunda.bpm.engine.RepositoryService;
import org.camunda.bpm.engine.repository.ProcessDefinition;
import org.camunda.bpm.model.bpmn.Bpmn;
import org.camunda.bpm.model.bpmn.BpmnModelInstance;
import org.camunda.bpm.model.bpmn.instance.camunda.CamundaProperty;

import java.util.Optional;

@Slf4j
public class ProcessDefinitionEncryptionParser {

    private RepositoryService repositoryService;

    public ProcessDefinitionEncryptionParser(RepositoryService repositoryService) {
        this.repositoryService = repositoryService;
    }

    public boolean shouldEncrypt(ProcessDefinition processDefinition, String encryptVariablesKey) {
        return isEncryptionEnabled(encryptVariablesKey, processDefinition);
    }

    public boolean shouldEncrypt(String processKey, String encryptVariablesKey) {
        ProcessDefinition processDefinition = this.repositoryService.createProcessDefinitionQuery()
                .processDefinitionKey(processKey)
                .latestVersion()
                .active()
                .singleResult();

        return isEncryptionEnabled(encryptVariablesKey, processDefinition);

    }

    private boolean isEncryptionEnabled(String encryptVariablesKey, ProcessDefinition processDefinition) {
        if (processDefinition == null) {
            throw new IllegalStateException("Process definition cannot be null");
        }

        BpmnModelInstance model = Bpmn.
                readModelFromStream(this.repositoryService.getProcessModel(processDefinition.getId()));

        Optional<CamundaProperty> hasEncryption = model.getModelElementsByType(CamundaProperty.class).stream()
                .filter(prop -> prop.getCamundaName().equalsIgnoreCase(encryptVariablesKey) &&
                        prop.getCamundaValue().equalsIgnoreCase("true")

                ).findFirst();

        return hasEncryption.isPresent();
    }
}
