package io.digitalpatterns.camunda.encryption;

import lombok.extern.slf4j.Slf4j;
import org.camunda.bpm.engine.delegate.VariableScope;
import org.camunda.bpm.engine.impl.cfg.ProcessEngineConfigurationImpl;
import org.camunda.bpm.engine.impl.scripting.ExecutableScript;
import org.camunda.bpm.engine.impl.scripting.env.ScriptingEnvironment;
import org.camunda.spin.Spin;
import org.codehaus.groovy.runtime.MethodClosure;
import org.springframework.util.CollectionUtils;

import javax.crypto.SealedObject;
import javax.script.Bindings;
import javax.script.ScriptEngine;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;

/**
 * This class enables the correct function construct to be used in the scripting environment.
 */

@Slf4j
public class EncryptionScriptingEnvironment extends ScriptingEnvironment {

    private static final Map<String, Map<String, Object>> TO_MERGE = new HashMap<>();

    private static final String JS_KEY = "javascript";
    private static final String GROOVY_KEY = "groovy";


    EncryptionScriptingEnvironment(ProcessEngineConfigurationImpl processEngineConfiguration,
                                   ProcessInstanceSpinVariableDecryptor processInstanceSpinVariableDecryptor,
                                   ProcessInstanceSpinVariableEncryptor processInstanceSpinVariableEncryptor
    ) {
        super(processEngineConfiguration.getScriptFactory(),
                processEngineConfiguration.getEnvScriptResolvers(),
                processEngineConfiguration.getScriptingEngines());
        registerFunctions(processInstanceSpinVariableDecryptor, processInstanceSpinVariableEncryptor);
    }


    private void registerFunctions(ProcessInstanceSpinVariableDecryptor processInstanceSpinVariableDecryptor,
                                   ProcessInstanceSpinVariableEncryptor processInstanceSpinVariableEncryptor) {
        TO_MERGE.put(JS_KEY,
                Map.of(EncryptionConstants.DECRYPT_KEY,
                        (Function<SealedObject, Spin>) processInstanceSpinVariableDecryptor::decrypt,
                        EncryptionConstants.ENCRYPT_KEY,
                        (Function<Object, SealedObject>) processInstanceSpinVariableEncryptor::encrypt)
        );

        TO_MERGE.put(GROOVY_KEY,
                Map.of(EncryptionConstants.DECRYPT_KEY,
                        new MethodClosure(processInstanceSpinVariableDecryptor, "decrypt"),
                        EncryptionConstants.ENCRYPT_KEY,
                        new MethodClosure(processInstanceSpinVariableEncryptor, "encrypt")));
    }


    @Override
    public Object execute(ExecutableScript script, VariableScope scope, Bindings bindings, ScriptEngine scriptEngine) {

        String language = scriptEngine.getFactory().getLanguageName().toLowerCase();
        Map<String, Object> bindingContext = new HashMap<>();
        switch (language) {
            case GROOVY_KEY:
                bindingContext.putAll(TO_MERGE.get(GROOVY_KEY));
                break;
            case JS_KEY:
            case "nashorn":
            case "js":
            case "ecmascript":
                bindingContext.putAll(TO_MERGE.get(JS_KEY));
                break;
            default:
                log.info("Unknown scripting language '{}'", language);
        }
        if (!CollectionUtils.isEmpty(bindingContext)) {
            bindings.putAll(bindingContext);
        }
        return super.execute(script, scope, bindings, scriptEngine);
    }
}
