/*
 * Decompiled with CFR 0.152.
 */
package io.digdag.storage.s3;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.HttpMethod;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.GeneratePresignedUrlRequest;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.amazonaws.services.s3.transfer.model.UploadResult;
import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.digdag.client.config.Config;
import io.digdag.spi.DirectDownloadHandle;
import io.digdag.spi.DirectUploadHandle;
import io.digdag.spi.Storage;
import io.digdag.spi.StorageFileNotFoundException;
import io.digdag.spi.StorageObject;
import io.digdag.spi.StorageObjectSummary;
import io.digdag.util.ResumableInputStream;
import io.digdag.util.RetryExecutor;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.time.Instant;
import java.util.Date;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3Storage
implements Storage {
    private static Logger logger = LoggerFactory.getLogger(S3Storage.class);
    private final Config config;
    private final AmazonS3Client client;
    private final String bucket;
    private final ExecutorService uploadExecutor;
    private final TransferManager transferManager;

    public S3Storage(Config config, AmazonS3Client client, String bucket) {
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)bucket) ? 1 : 0) != 0, (Object)"bucket is null or empty");
        this.config = config;
        this.client = client;
        this.bucket = bucket;
        this.uploadExecutor = Executors.newCachedThreadPool(new ThreadFactoryBuilder().setNameFormat("storage-s3-upload-transfer-%d").build());
        this.transferManager = new TransferManager((AmazonS3)client, this.uploadExecutor);
    }

    private RetryExecutor uploadRetryExecutor() {
        return RetryExecutor.retryExecutor();
    }

    private RetryExecutor getRetryExecutor() {
        return RetryExecutor.retryExecutor();
    }

    public StorageObject open(String key) throws StorageFileNotFoundException {
        Preconditions.checkArgument((key != null ? 1 : 0) != 0, (Object)"key is null");
        String errorMessage = "opening file bucket " + this.bucket + " key " + key;
        GetObjectRequest req = new GetObjectRequest(this.bucket, key);
        S3Object obj = this.getWithRetry(errorMessage, () -> this.client.getObject(req));
        long actualSize = obj.getObjectMetadata().getContentLength();
        InputStream stream = this.overrideCloseToAbort(obj.getObjectContent());
        ResumableInputStream resumable = new ResumableInputStream(stream, (offset, closedCause) -> {
            try {
                S3ObjectInputStream raw = this.getWithRetry(errorMessage, () -> {
                    req.setRange(offset, actualSize - offset - 1L);
                    return this.client.getObject(req);
                }).getObjectContent();
                return this.overrideCloseToAbort(raw);
            }
            catch (StorageFileNotFoundException ex) {
                throw new IOException(ex);
            }
        });
        return new StorageObject((InputStream)resumable, actualSize);
    }

    private InputStream overrideCloseToAbort(final S3ObjectInputStream raw) {
        return new FilterInputStream((InputStream)raw){

            @Override
            public void close() throws IOException {
                raw.abort();
            }
        };
    }

    public String put(String key, long contentLength, Storage.UploadStreamProvider payload) throws IOException {
        Preconditions.checkArgument((key != null ? 1 : 0) != 0, (Object)"key is null");
        ObjectMetadata meta = new ObjectMetadata();
        meta.setContentLength(contentLength);
        try {
            return (String)this.uploadRetryExecutor().onRetry((exception, retryCount, retryLimit, retryWait) -> logger.warn("Retrying uploading file bucket " + this.bucket + " key " + key + " error: " + exception)).retryIf(exception -> !(exception instanceof IOException) && !(exception instanceof InterruptedException)).runInterruptible(() -> {
                try (InputStream in = payload.open();){
                    PutObjectRequest req = new PutObjectRequest(this.bucket, key, in, meta);
                    UploadResult result = this.transferManager.upload(req).waitForUploadResult();
                    String string = result.getETag();
                    return string;
                }
            });
        }
        catch (InterruptedException ex) {
            throw Throwables.propagate((Throwable)ex);
        }
        catch (RetryExecutor.RetryGiveupException ex) {
            Exception cause = ex.getCause();
            Throwables.propagateIfInstanceOf((Throwable)cause, IOException.class);
            throw Throwables.propagate((Throwable)cause);
        }
    }

    public void list(String keyPrefix, Storage.FileListing callback) {
        ObjectListing listing;
        Preconditions.checkArgument((keyPrefix != null ? 1 : 0) != 0, (Object)"keyPrefix is null");
        String errorMessage = "listing files on bucket " + this.bucket + " prefix " + keyPrefix;
        ListObjectsRequest req = new ListObjectsRequest();
        req.setBucketName(this.bucket);
        req.setPrefix(keyPrefix);
        do {
            try {
                listing = this.getWithRetry(errorMessage, () -> this.client.listObjects(req));
            }
            catch (StorageFileNotFoundException ex) {
                throw Throwables.propagate((Throwable)ex.getCause());
            }
            callback.accept(Lists.transform((List)listing.getObjectSummaries(), summary -> StorageObjectSummary.builder().key(summary.getKey()).contentLength(summary.getSize()).lastModified(summary.getLastModified().toInstant()).build()));
            req.setMarker(listing.getNextMarker());
        } while (listing.isTruncated());
    }

    public Optional<DirectDownloadHandle> getDirectDownloadHandle(String key) {
        long secondsToExpire = (Long)this.config.get("direct_download_expiration", Long.class, (Object)600L);
        GeneratePresignedUrlRequest req = new GeneratePresignedUrlRequest(this.bucket, key);
        req.setExpiration(Date.from(Instant.now().plusSeconds(secondsToExpire)));
        String url = this.client.generatePresignedUrl(req).toString();
        return Optional.of((Object)DirectDownloadHandle.of((String)url));
    }

    public Optional<DirectUploadHandle> getDirectUploadHandle(String key) {
        long secondsToExpire = (Long)this.config.get("direct_upload_expiration", Long.class, (Object)600L);
        GeneratePresignedUrlRequest req = new GeneratePresignedUrlRequest(this.bucket, key);
        req.setMethod(HttpMethod.PUT);
        req.setExpiration(Date.from(Instant.now().plusSeconds(secondsToExpire)));
        String url = this.client.generatePresignedUrl(req).toString();
        return Optional.of((Object)DirectUploadHandle.of((String)url));
    }

    private <T> T getWithRetry(String message, Callable<T> callable) throws StorageFileNotFoundException {
        try {
            return (T)this.getRetryExecutor().onRetry((exception, retryCount, retryLimit, retryWait) -> logger.warn(String.format("Retrying %s (%d/%d): %s", message, retryCount, retryLimit, exception))).retryIf(exception -> !S3Storage.isNotFoundException(exception)).runInterruptible(() -> callable.call());
        }
        catch (InterruptedException ex) {
            throw Throwables.propagate((Throwable)ex);
        }
        catch (RetryExecutor.RetryGiveupException ex) {
            Exception cause = ex.getCause();
            if (S3Storage.isNotFoundException(cause)) {
                throw new StorageFileNotFoundException("S3 file not found", (Throwable)cause);
            }
            throw Throwables.propagate((Throwable)cause);
        }
    }

    private static boolean isNotFoundException(Exception ex) {
        return ex instanceof AmazonServiceException && ((AmazonServiceException)ex).getStatusCode() == 404;
    }
}

