/*
 * Decompiled with CFR 0.152.
 */
package io.deepstream;

import com.google.gson.JsonElement;
import io.deepstream.ConnectionStateListener;
import io.deepstream.DeepstreamClient;
import io.deepstream.DeepstreamConfig;
import io.deepstream.DeepstreamLoginException;
import io.deepstream.Endpoint;
import io.deepstream.EndpointTCP;
import io.deepstream.IConnection;
import io.deepstream.Message;
import io.deepstream.MessageBuilder;
import io.deepstream.MessageParser;
import io.deepstream.constants.Actions;
import io.deepstream.constants.ConnectionState;
import io.deepstream.constants.EndpointType;
import io.deepstream.constants.Event;
import io.deepstream.constants.Topic;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

class Connection
implements IConnection {
    private final DeepstreamClient client;
    private final String originalUrl;
    private final ArrayList<ConnectionStateListener> connectStateListeners;
    private final DeepstreamConfig options;
    private Endpoint endpoint;
    private boolean tooManyAuthAttempts;
    private boolean challengeDenied;
    private boolean deliberateClose;
    private boolean redirecting;
    private Timer reconnectTimeout;
    private int reconnectionAttempt;
    private StringBuilder messageBuffer;
    private String url;
    private ConnectionState connectionState;
    private DeepstreamClient.LoginCallback loginCallback;
    private JsonElement authParameters;
    private ExecutorService rpcThread;
    private ExecutorService recordThread;
    private ExecutorService eventThread;

    Connection(String url, DeepstreamConfig options, DeepstreamClient client) throws URISyntaxException {
        this(url, options, client, null);
        this.endpoint = this.createEndpoint();
    }

    Connection(String url, DeepstreamConfig options, DeepstreamClient client, Endpoint endpoint) {
        this.client = client;
        this.connectStateListeners = new ArrayList();
        this.originalUrl = url;
        this.url = url;
        this.connectionState = ConnectionState.CLOSED;
        this.messageBuffer = new StringBuilder();
        this.tooManyAuthAttempts = false;
        this.challengeDenied = false;
        this.deliberateClose = false;
        this.redirecting = false;
        this.reconnectTimeout = null;
        this.reconnectionAttempt = 0;
        this.options = options;
        this.endpoint = endpoint;
        this.recordThread = Executors.newSingleThreadExecutor();
        this.eventThread = Executors.newSingleThreadExecutor();
        this.rpcThread = Executors.newSingleThreadExecutor();
    }

    void authenticate(JsonElement authParameters, DeepstreamClient.LoginCallback loginCallback) throws DeepstreamLoginException {
        if (this.tooManyAuthAttempts || this.challengeDenied) {
            this.client.onError(Topic.ERROR, Event.IS_CLOSED, "The client's connection was closed");
            return;
        }
        this.loginCallback = loginCallback;
        this.authParameters = authParameters;
        if (this.connectionState == ConnectionState.AWAITING_AUTHENTICATION) {
            this.sendAuthMessage();
        }
    }

    @Override
    public void send(String message) {
        if (this.connectionState != ConnectionState.OPEN) {
            this.messageBuffer.append(message);
        } else {
            this.endpoint.send(message);
        }
    }

    @Override
    public void sendMsg(Topic topic, Actions action, String[] data) {
        this.send(MessageBuilder.getMsg(topic, action, data));
    }

    private void sendAuthMessage() {
        this.setState(ConnectionState.AUTHENTICATING);
        String authMessage = MessageBuilder.getMsg(Topic.AUTH, Actions.REQUEST, this.authParameters.toString());
        this.endpoint.send(authMessage);
    }

    void addConnectionChangeListener(ConnectionStateListener connectionStateListener) {
        this.connectStateListeners.add(connectionStateListener);
    }

    void removeConnectionChangeListener(ConnectionStateListener connectionStateListener) {
        this.connectStateListeners.remove(connectionStateListener);
    }

    ConnectionState getConnectionState() {
        return this.connectionState;
    }

    public void close() {
        this.deliberateClose = true;
        if (this.endpoint != null) {
            this.endpoint.close();
            this.endpoint = null;
        }
        if (this.reconnectTimeout != null) {
            this.reconnectTimeout.cancel();
            this.reconnectTimeout = null;
        }
    }

    void onOpen() {
        this.setState(ConnectionState.AWAITING_CONNECTION);
    }

    void onError(final String error) {
        this.setState(ConnectionState.ERROR);
        Timer timer = new Timer();
        timer.schedule(new TimerTask(){

            @Override
            public void run() {
                Connection.this.client.onError(null, Event.CONNECTION_ERROR, error);
            }
        }, 1000L);
    }

    void onMessage(String rawMessage) {
        List<Message> parsedMessages = MessageParser.parse(rawMessage, this.client);
        for (final Message message : parsedMessages) {
            if (message.topic == Topic.CONNECTION) {
                this.handleConnectionResponse(message);
                continue;
            }
            if (message.topic == Topic.AUTH) {
                this.handleAuthResponse(message);
                continue;
            }
            if (message.topic == Topic.EVENT) {
                this.eventThread.execute(new Runnable(){

                    @Override
                    public void run() {
                        ((Connection)Connection.this).client.event.handle(message);
                    }
                });
                continue;
            }
            if (message.topic == Topic.RPC) {
                this.rpcThread.execute(new Runnable(){

                    @Override
                    public void run() {
                        ((Connection)Connection.this).client.rpc.handle(message);
                    }
                });
                continue;
            }
            if (message.topic != Topic.RECORD) continue;
            this.recordThread.execute(new Runnable(){

                @Override
                public void run() {
                    ((Connection)Connection.this).client.record.handle(message);
                }
            });
        }
    }

    void onClose() throws URISyntaxException {
        if (this.redirecting) {
            this.redirecting = false;
            this.createEndpoint();
        } else if (this.deliberateClose) {
            this.setState(ConnectionState.CLOSED);
        } else {
            if (!this.originalUrl.equals(this.url)) {
                this.url = this.originalUrl;
                this.createEndpoint();
                return;
            }
            this.tryReconnect();
        }
    }

    private void handleConnectionResponse(Message message) {
        if (message.action == Actions.ACK) {
            this.setState(ConnectionState.AWAITING_AUTHENTICATION);
            if (this.authParameters != null) {
                this.sendAuthMessage();
            }
        } else if (message.action == Actions.CHALLENGE) {
            this.setState(ConnectionState.CHALLENGING);
            this.endpoint.send(MessageBuilder.getMsg(Topic.CONNECTION, Actions.CHALLENGE_RESPONSE, this.originalUrl));
        } else if (message.action == Actions.REJECTION) {
            this.challengeDenied = true;
            this.close();
        } else if (message.action == Actions.REDIRECT) {
            this.url = message.data[0];
            this.redirecting = true;
            this.endpoint.close();
        }
    }

    private void handleAuthResponse(Message message) {
        if (message.action == Actions.ERROR) {
            if (message.data[0].equals(Event.TOO_MANY_AUTH_ATTEMPTS.name())) {
                this.deliberateClose = true;
                this.tooManyAuthAttempts = true;
            } else {
                this.setState(ConnectionState.AWAITING_AUTHENTICATION);
            }
            if (this.loginCallback != null) {
                this.loginCallback.loginFailed(Event.getEvent(message.data[0]), MessageParser.convertTyped(message.data[1], this.client));
            }
        } else if (message.action == Actions.ACK) {
            this.setState(ConnectionState.OPEN);
            if (this.messageBuffer.length() > 0) {
                this.endpoint.send(this.messageBuffer.toString());
                this.messageBuffer = new StringBuilder();
            }
            if (this.loginCallback != null) {
                this.loginCallback.loginSuccess(new HashMap());
            }
        }
    }

    private void setState(ConnectionState connectionState) {
        this.connectionState = connectionState;
        if (connectionState == ConnectionState.AWAITING_CONNECTION && this.authParameters != null) {
            this.sendAuthMessage();
        }
        for (ConnectionStateListener connectStateListener : this.connectStateListeners) {
            connectStateListener.connectionStateChanged(connectionState);
        }
    }

    private Endpoint createEndpoint() throws URISyntaxException {
        EndpointTCP endpoint = null;
        if (this.options.getEndpointType().equals((Object)EndpointType.TCP)) {
            endpoint = new EndpointTCP(this.url, this.options, this);
            this.endpoint = endpoint;
        } else if (this.options.getEndpointType().equals((Object)EndpointType.ENGINEIO)) {
            System.out.println("EngineIO doesn't transpile");
        }
        return endpoint;
    }

    private void tryReconnect() {
        if (this.reconnectTimeout != null) {
            return;
        }
        int maxReconnectAttempts = this.options.getMaxReconnectAttempts();
        int reconnectIntervalIncrement = this.options.getReconnectIntervalIncrement();
        int maxReconnectInterval = this.options.getMaxReconnectInterval();
        if (this.reconnectionAttempt < maxReconnectAttempts) {
            this.setState(ConnectionState.RECONNECTING);
            this.reconnectTimeout = new Timer();
            this.reconnectTimeout.schedule(new TimerTask(){

                @Override
                public void run() {
                    Connection.this.tryOpen();
                }
            }, Math.min(reconnectIntervalIncrement * this.reconnectionAttempt, maxReconnectInterval));
            ++this.reconnectionAttempt;
        } else {
            this.clearReconnect();
            this.close();
        }
    }

    private void tryOpen() {
        this.reconnectTimeout.cancel();
        this.reconnectTimeout = null;
        this.endpoint.open();
    }

    private void clearReconnect() {
        this.reconnectTimeout = null;
        this.reconnectionAttempt = 0;
    }
}

