/*
 *  Copyright 2017 The WebRTC project authors. All Rights Reserved.
 *
 *  Use of this source code is governed by a BSD-style license
 *  that can be found in the LICENSE file in the root of the source
 *  tree. An additional intellectual property rights grant can be found
 *  in the file PATENTS.  All contributing project authors may
 *  be found in the AUTHORS file in the root of the source tree.
 */

package org.webrtc;

import android.media.MediaCodecInfo;
import android.support.annotation.Nullable;
import java.util.Arrays;
import java.util.LinkedHashSet;

/** Helper class that combines HW and SW encoders. */
public class DefaultVideoEncoderFactory implements VideoEncoderFactory {
  private final VideoEncoderFactory hardwareVideoEncoderFactory;
  private final VideoEncoderFactory softwareVideoEncoderFactory = new SoftwareVideoEncoderFactory();
  private final @Nullable VideoEncoderFactory platformSoftwareVideoEncoderFactory;

  /** Create encoder factory using default hardware encoder factory. */
  public DefaultVideoEncoderFactory(
      EglBase.Context eglContext, boolean enableIntelVp8Encoder, boolean enableH264HighProfile) {
    this(eglContext, enableIntelVp8Encoder, enableH264HighProfile, null);
  }

  public DefaultVideoEncoderFactory(
          EglBase.Context eglContext, boolean enableIntelVp8Encoder, boolean enableH264HighProfile, @Nullable Predicate<MediaCodecInfo> codecAllowedPredicate) {
    this.hardwareVideoEncoderFactory =
            new HardwareVideoEncoderFactory(eglContext, enableIntelVp8Encoder, enableH264HighProfile, codecAllowedPredicate);
    this.platformSoftwareVideoEncoderFactory = new PlatformSoftwareVideoEncoderFactory(eglContext);
  }

  /** Create encoder factory using explicit hardware encoder factory. */
  DefaultVideoEncoderFactory(VideoEncoderFactory hardwareVideoEncoderFactory) {
    this.hardwareVideoEncoderFactory = hardwareVideoEncoderFactory;
    this.platformSoftwareVideoEncoderFactory = null;
  }

  @Nullable
  @Override
  public VideoEncoder createEncoder(VideoCodecInfo info) {
    VideoEncoder softwareEncoder = softwareVideoEncoderFactory.createEncoder(info);
    final VideoEncoder hardwareEncoder = hardwareVideoEncoderFactory.createEncoder(info);
    if (softwareEncoder == null && platformSoftwareVideoEncoderFactory != null) {
      softwareEncoder = platformSoftwareVideoEncoderFactory.createEncoder(info);
    }
    if (hardwareEncoder != null && softwareEncoder != null) {
      // Both hardware and software supported, wrap it in a software fallback
      return new VideoEncoderFallback(
          /* fallback= */ softwareEncoder, /* primary= */ hardwareEncoder);
    }
    return hardwareEncoder != null ? hardwareEncoder : softwareEncoder;
  }

  @Override
  public VideoCodecInfo[] getSupportedCodecs() {
    LinkedHashSet<VideoCodecInfo> supportedCodecInfos = new LinkedHashSet<VideoCodecInfo>();

    supportedCodecInfos.addAll(Arrays.asList(softwareVideoEncoderFactory.getSupportedCodecs()));
    supportedCodecInfos.addAll(Arrays.asList(hardwareVideoEncoderFactory.getSupportedCodecs()));
    if (platformSoftwareVideoEncoderFactory != null) {
      supportedCodecInfos.addAll(
              Arrays.asList(platformSoftwareVideoEncoderFactory.getSupportedCodecs()));
    }

    return supportedCodecInfos.toArray(new VideoCodecInfo[supportedCodecInfos.size()]);
  }
  /** Factory for Android platform software VideoDecoders. */
  private static class PlatformSoftwareVideoEncoderFactory extends HardwareVideoEncoderFactory {
    /**
    * Creates a PlatformSoftwareVideoDecoderFactory that supports surface texture rendering.
    *
    * @param sharedContext The textures generated will be accessible from this context. May be null,
    *                      this disables texture support.
    */
    public PlatformSoftwareVideoEncoderFactory(@Nullable EglBase.Context sharedContext) {
      super(sharedContext, false, false, arg -> {
        final String name = arg.getName();
        for (String prefix : MediaCodecUtils.SOFTWARE_IMPLEMENTATION_PREFIXES) {
          if (name.startsWith(prefix)) {
            return true;
          }
        }
        return false;
      });
    }
  }
}
