package io.ciogram.types

import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonContentPolymorphicSerializer
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.jsonObject

@Serializable(with = InputMessageContentSerializer::class)
public sealed class InputMessageContent

internal object InputMessageContentSerializer : JsonContentPolymorphicSerializer<InputMessageContent>(InputMessageContent::class) {
    override fun selectDeserializer(element: JsonElement): DeserializationStrategy<out InputMessageContent> = when {
        "message_text" in element.jsonObject -> InputTextMessageContent.serializer()
        "latitude" in element.jsonObject && "longitude" in element.jsonObject -> if ("title" in element.jsonObject && "address" in element.jsonObject) InputVenueMessageContent.serializer() else InputLocationMessageContent.serializer()
        "phone_number" in element.jsonObject -> InputContactMessageContent.serializer()
        "payload" in element.jsonObject -> InputInvoiceMessageContent.serializer()
        else -> throw IllegalArgumentException("Input message content is not specified")
    }
}

@Serializable
public data class InputTextMessageContent(
    @SerialName("message_text") val messageText: String,
    @SerialName("parse_mode") val parseMode: String? = null,
    val entities: List<MessageEntity>? = null,
    @SerialName("disable_web_page_preview") val disableWebPagePreview: Boolean? = null
) : InputMessageContent()

@Serializable
public data class InputLocationMessageContent(
    val latitude: Float,
    val longitude: Float,
    @SerialName("horizontal_accuracy") val horizontalAccuracy: Float? = null,
    @SerialName("live_period") val livePeriod: Int? = null,
    val heading: Int? = null,
    val proximityAlertRadius: Int? = null
) : InputMessageContent()

@Serializable
public data class InputVenueMessageContent(
    val latitude: Float,
    val longitude: Float,
    val title: String,
    val address: String,
    @SerialName("foursquare_id") val foursquareId: String? = null,
    @SerialName("foursquare_type") val foursquareType: String? = null,
    @SerialName("google_place_id") val googlePlaceId: String? = null,
    @SerialName("google_place_type") val googlePlaceType: String? = null
) : InputMessageContent()

@Serializable
public data class InputContactMessageContent(
    @SerialName("phone_number") val phoneNumber: String,
    @SerialName("first_name") val firstName: String,
    @SerialName("last_name") val lastName: String? = null,
    val vcard: String? = null
) : InputMessageContent()

@Serializable
public data class InputInvoiceMessageContent(
    val title: String,
    val description: String,
    val payload: String,
    @SerialName("provider_token") val providerToken: String,
    val currency: String,
    val prices: List<LabeledPrice>,
    @SerialName("max_tip_amount") val maxTipAmount: Int? = null,
    @SerialName("suggested_tip_amounts") val suggestedTipAmounts: List<Int>? = null,
    @SerialName("provider_data") val providerData: String? = null,
    @SerialName("photo_url") val photoUrl: String? = null,
    @SerialName("photo_size") val photoSize: Int? = null,
    @SerialName("photo_width") val photoWidth: Int? = null,
    @SerialName("photo_height") val photoHeight: Int? = null,
    @SerialName("need_name") val needName: Boolean? = null,
    @SerialName("need_phone_number") val needPhoneNumber: Boolean? = null,
    @SerialName("need_shipping_address") val needShippingAddress: Boolean? = null,
    @SerialName("send_phone_number_to_provider") val sendPhoneNumberToProvider: Boolean? = null,
    @SerialName("send_email_to_provider") val sendEmailToProvider: Boolean? = null,
    @SerialName("is_flexible") val isFlexible: Boolean? = null
) : InputMessageContent()