/**
 * Copyright © 2016 Thomas Biesaart (thomas.biesaart@gmail.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.chapp.commons.locale;

import javax.annotation.processing.Messager;
import javax.lang.model.element.Element;
import javax.tools.Diagnostic;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.Format;
import java.text.MessageFormat;
import java.text.NumberFormat;
import java.util.*;

/**
 * This class contains all required information about a class annotated with {@link Bundle}.
 * It acts as a helper class for the {@link BundleProcessor} and extracts all the required information
 * for the template.
 *
 * @author Thomas Biesaart
 */
class BundleValidator {
    private final Bundle annotation;
    private final Element annotatedElement;
    private final Messager messager;
    private Boolean valid;
    private Map<String, List<String>> parameterTypes = new HashMap<String, List<String>>();

    BundleValidator(Bundle annotation, Element annotatedElement, Messager messager) {
        this.annotation = annotation;
        this.annotatedElement = annotatedElement;
        this.messager = messager;
    }

    public boolean isValid() {
        if (valid != null) {
            return valid;
        }

        valid = validateDefaultBundle() &&
                validateRequiredLocales();

        return valid;
    }

    private boolean validateDefaultBundle() {
        return validateBundle(null);
    }

    private boolean validateRequiredLocales() {
        boolean success = true;
        for (String locale : annotation.requiredLocales()) {
            if (!validateBundle(locale)) {
                success = false;
            }
        }
        return success;
    }

    private boolean validateBundle(String localeName) {
        String locale = localeName == null ? "default" : localeName;
        String bundlePath = getBundleName();
        if (localeName != null) {
            bundlePath += "_" + locale;
        }
        String resourcePath = "/" + bundlePath + ".properties";
        InputStream resourceBundle = getClass().getResourceAsStream(resourcePath);
        if (resourceBundle == null) {
            error("Missing locale '" + locale + "' for resource bundle '" + getBundleName() + "'");
            return false;
        } else {
            Properties properties = new Properties();
            try {
                try {
                    properties.load(resourceBundle);
                } finally {
                    resourceBundle.close();
                }
            } catch (IOException e) {
                error("Invalid Format for resource bundle '" + getBundleName() + "' with locale '" + locale + "': " + e.getMessage());
                return false;
            }
            return validateBundle(properties, locale, localeName == null);
        }
    }

    private boolean validateBundle(Properties properties, String locale, boolean storeFormats) {
        boolean success = true;
        for (Map.Entry<Object, Object> entry : getDefaultEntries().entrySet()) {
            String key = entry.getKey().toString();
            if (!isEntryPresent(properties, key, locale) ||
                    !isEntryValid(key, properties.getProperty(key), locale, storeFormats)) {
                success = false;
            }
        }
        return success;
    }

    private boolean isEntryValid(String key, String message, String locale, boolean storeFormats) {
        try {
            MessageFormat messageFormat = new MessageFormat(message);
            List<String> types = new ArrayList<String>();
            for (Format format : messageFormat.getFormatsByArgumentIndex()) {
                types.add(getType(format));
            }

            if (storeFormats) {
                // Save these formats
                this.parameterTypes.put(key, types);
            } else {
                if (!areParametersValid(this.parameterTypes.get(key), types, locale, key)) {
                    return false;
                }
            }
        } catch (IllegalArgumentException e) {
            error(
                    "Invalid pattern for entry '" + key + "' in resource bundle '" + getBundleName() +
                            "' with locale '" + locale + "': " +
                            e.getLocalizedMessage()
            );
            return false;
        }
        return true;
    }

    private boolean areParametersValid(List<String> expectedTypes, List<String> realTypes, String locale, String key) {
        if (expectedTypes.size() != realTypes.size()) {
            error(
                    "Parameter count mismatch in message '" + key + "' of resource bundle '" + getBundleName() + "'.\n" +
                            "The '" + locale + "' expects " + realTypes.size() +
                            " parameters but the default locale expects " + expectedTypes.size() + " parameters."
            );
            return false;
        } else {
            for (int i = 0; i < expectedTypes.size(); i++) {
                String actual = realTypes.get(i);
                String expected = expectedTypes.get(i);

                if (!actual.equalsIgnoreCase(expected)) {
                    error(
                            "Parameter count mismatch in message '" + key + "' of resource bundle '" + getBundleName() + "'.\n" +
                                    "Default locale: " + expected + "\n" +
                                    locale + ": " + actual
                    );
                }
            }
            return true;
        }
    }

    private boolean isEntryPresent(Properties properties, String key, String locale) {
        if (!properties.containsKey(key)) {
            error("Missing entry '" + key + "' for required locale '" + locale + "' for resource bundle '" + getBundleName() + "'");
            return false;
        }
        return true;
    }

    private String getType(Format format) {
        if (format instanceof DateFormat) {
            return "Date";
        } else if (format instanceof NumberFormat) {
            return "Number";
        } else {
            return "Object";
        }
    }

    public String getBundleName() {
        return annotation.bundleName();
    }

    public String getTargetQualifiedClassName() {
        return annotation.className();
    }

    public String getTargetPackage() {
        int index = annotation.className().lastIndexOf('.');
        if (index == -1) {
            return null;
        }
        return annotation.className().substring(0, index);
    }

    public Map<String, List<String>> getParameterTypes() {
        return parameterTypes;
    }

    public String getTargetClassName() {
        int index = annotation.className().lastIndexOf('.');
        return annotation.className().substring(index + 1);
    }

    public void info(String message) {
        messager.printMessage(Diagnostic.Kind.NOTE, message, annotatedElement);
    }

    public void error(String message) {
        error(message, annotatedElement);
    }

    public void error(String message, Element element) {
        messager.printMessage(Diagnostic.Kind.ERROR, message, element);
    }

    public Properties getDefaultEntries() {
        Properties entries = new Properties();
        InputStream stream = getClass().getResourceAsStream("/" + getBundleName() + ".properties");
        if (stream == null) {
            error("No default resource bundle found for name '" + getBundleName() + "'");
        } else {
            try {
                try {
                    entries.load(stream);
                } finally {
                    stream.close();
                }
            } catch (IOException e) {
                error("Invalid contents for resource bundle: " + e.getMessage());
            }
        }
        return entries;
    }


}
