/**
 * Copyright © 2016 Thomas Biesaart (thomas.biesaart@gmail.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.chapp.commons.locale;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedAnnotationTypes;
import javax.annotation.processing.SupportedSourceVersion;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.Element;
import javax.lang.model.element.TypeElement;
import javax.tools.JavaFileObject;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.*;

/**
 * This class is responsible for processing the {@link Bundle} annotation.
 * It will make use of the helper class {@link BundleValidator} to extract the required information
 * and generate the designated class.
 *
 * @author Thomas Biesaart
 */
@SupportedAnnotationTypes({"io.chapp.commons.locale.Bundle", "io.chapp.commons.locale.Bundles"})
@SupportedSourceVersion(SourceVersion.RELEASE_6)
public class BundleProcessor extends AbstractProcessor {
    private final Template template;

    public BundleProcessor() throws IOException {
        this(new Configuration(Configuration.VERSION_2_3_23));
    }

    public BundleProcessor(Configuration configuration) throws IOException {
        configuration.setClassForTemplateLoading(BundleProcessor.class, "/io/chapp/commons/locale/");
        this.template = configuration.getTemplate("localized-class.ftl");
    }

    /**
     * This method will convert the input string to UpperCamelCase provided that they consist of parts that
     * are separated by periods, underscores or dashes.
     *
     * @param input the input string
     * @return the UCC string
     */
    private static String toUpperCamelCase(String input) {
        String[] parts = input.split("[._-]");
        StringBuilder builder = new StringBuilder(input.length() - parts.length + 1);
        for (String part : parts) {
            builder.append(Character.toUpperCase(part.charAt(0))).append(part.substring(1).toLowerCase());
        }
        return builder.toString();
    }

    @Override
    public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
        for (Element annotatedElement : roundEnv.getElementsAnnotatedWith(Bundle.class)) {
            process(annotatedElement.getAnnotation(Bundle.class), annotatedElement);
        }
        for (Element annotatedElement : roundEnv.getElementsAnnotatedWith(Bundles.class)) {
            for (Bundle bundle : annotatedElement.getAnnotation(Bundles.class).value()) {
                process(bundle, annotatedElement);
            }
        }
        return false;
    }

    private void process(Bundle resourceBundle, Element annotatedElement) {
        BundleValidator validator = new BundleValidator(resourceBundle, annotatedElement, processingEnv.getMessager());

        try {
            if (validator.isValid()) {
                process(validator);
            }
        } catch (IOException e) {
            validator.error(e.getMessage());
        }
    }

    private void process(BundleValidator classData) throws IOException {
        classData.info("Generating Localization class for " + classData.getBundleName());

        JavaFileObject object = processingEnv.getFiler().createSourceFile(
                classData.getTargetQualifiedClassName()
        );

        Map<String, Object> model = buildModel(classData);

        PrintWriter writer = new PrintWriter(object.openWriter());
        try {
            template.process(model, writer);
        } catch (TemplateException e) {
            throw new IOException("Template Error: " + e.getMessage(), e);
        } finally {
            writer.close();
        }
    }

    private Map<String, Object> buildModel(BundleValidator classData) {
        Map<String, Object> model = new HashMap<String, Object>();

        // Get the package name
        if (classData.getTargetPackage() != null) {
            model.put("package", classData.getTargetPackage());
        }
        model.put("className", classData.getTargetClassName());
        model.put("bundleName", classData.getBundleName());

        List<Map<String, Object>> resources = new ArrayList<Map<String, Object>>();

        for (Object key : classData.getDefaultEntries().keySet()) {
            Map<String, Object> resource = new HashMap<String, Object>();
            resource.put("key", key.toString());
            resource.put("methodName",
                    "get" + toUpperCamelCase(key.toString())
            );

            resource.put("parameterTypes", classData.getParameterTypes().get(key.toString()));

            resources.add(resource);
        }

        model.put("resources", resources);
        return model;
    }
}
