package io.cana.cella.common.serialize;

import io.cana.steam.common.digest.Base64;
import io.cana.steam.common.utils.StringUtil;
import io.protostuff.LinkedBuffer;
import io.protostuff.MessageCollectionSchema;
import io.protostuff.MessageMapSchema;
import io.protostuff.ProtostuffIOUtil;
import io.protostuff.Schema;
import io.protostuff.runtime.RuntimeSchema;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author Dasheng
 * @create 2020-03-03 10:32
 */
public class Protostuff {
    public static class Inner {
        private static final ThreadLocal<LinkedBuffer> BUFFER = ThreadLocal.withInitial(() -> LinkedBuffer.allocate(1024 * 1024));
        private static final Map<Class<?>, Schema<?>> SCHEMA_CACHE = new ConcurrentHashMap<>();

        @SuppressWarnings("unchecked")
        private static <T> Schema<T> getSchema(Class<T> clazz) {
            Schema<T> schema = (Schema<T>) SCHEMA_CACHE.get(clazz);
            if (Objects.isNull(schema)) {
                schema = RuntimeSchema.getSchema(clazz);
                if (Objects.nonNull(schema)) {
                    SCHEMA_CACHE.put(clazz, schema);
                }
            }
            return schema;
        }
    }

    public static LinkedBuffer getBuffer() {
        return Inner.BUFFER.get();
    }

    public static byte[] collection2Byte(Collection<String> list) {
        return collection2Byte(list, String.class);
    }

    public static <E> byte[] collection2Byte(Collection<E> list, Class<E> ele) {
        MessageCollectionSchema<E> schema = new MessageCollectionSchema<>(Inner.getSchema(ele));
        try {
            return ProtostuffIOUtil.toByteArray(list, schema, getBuffer());
        } finally {
            getBuffer().clear();
        }
    }

    public static String collection2String(Collection<String> list) {
        return Base64.encode(collection2Byte(list));
    }

    public static <E> String collection2String(Collection<E> list, Class<E> ele) {
        return Base64.encode(collection2Byte(list, ele));
    }

    public static <V> byte[] map2Byte(Map<String, V> map, Class<V> value) {
        return map2Byte(map, String.class, value);
    }

    public static <K, V> byte[] map2Byte(Map<K, V> map, Class<K> key, Class<V> value) {
        MessageMapSchema<K, V> schema = new MessageMapSchema<>(Inner.getSchema(key), Inner.getSchema(value));
        try {
            return ProtostuffIOUtil.toByteArray(map, schema, getBuffer());
        } finally {
            getBuffer().clear();
        }
    }

    public static <V> String map2String(Map<String, V> map, Class<V> value) {
        return Base64.encode(map2Byte(map, String.class, value));
    }

    public static <K, V> String map2String(Map<K, V> map, Class<K> key, Class<V> value) {
        return Base64.encode(map2Byte(map, key, value));
    }

    @SuppressWarnings("unchecked")
    public static <T> byte[] toByte(T obj) {
        try {
            Class<T> clazz = (Class<T>) obj.getClass();
            return ProtostuffIOUtil.toByteArray(obj, Inner.getSchema(clazz), getBuffer());
        } finally {
            getBuffer().clear();
        }
    }

    public static Collection<String> byte2Collection(byte[] data) {
        return byte2Collection(data, String.class);
    }

    public static <E> Collection<E> byte2Collection(byte[] data, Class<E> ele) {
        MessageCollectionSchema<E> schema = new MessageCollectionSchema<>(Inner.getSchema(ele));
        Collection<E> list = schema.newMessage();
        ProtostuffIOUtil.mergeFrom(data, list, schema);
        return list;
    }

    public static Collection<String> string2Collection(String data) {
        return byte2Collection(Base64.decode(data), String.class);
    }

    public static <E> Collection<E> string2Collection(String data, Class<E> ele) {
        return byte2Collection(Base64.decode(data), ele);
    }

    public static Map<String, String> byte2Map(byte[] data) {
        return byte2Map(data, String.class, String.class);
    }

    public static <V> Map<String, V> byte2Map(byte[] data, Class<V> value) {
        return byte2Map(data, String.class, value);
    }

    public static <K, V> Map<K, V> byte2Map(byte[] data, Class<K> key, Class<V> value) {
        MessageMapSchema<K, V> schema = new MessageMapSchema<>(Inner.getSchema(key), Inner.getSchema(value));
        Map<K, V> map = schema.newMessage();
        ProtostuffIOUtil.mergeFrom(data, map, schema);
        return map;
    }

    public static <V> Map<String, V> string2Map(String data, Class<V> value) {
        return byte2Map(Base64.decode(data), String.class, value);
    }

    public static <K, V> Map<K, V> string2Map(String data, Class<K> key, Class<V> value) {
        return byte2Map(Base64.decode(data), key, value);
    }

    public static <T> T fromString(String str, Class<T> clazz) {
        if (StringUtil.isBlank(str)) {
            return null;
        }
        Schema<T> schema = Inner.getSchema(clazz);
        T obj = schema.newMessage();
        ProtostuffIOUtil.mergeFrom(Base64.decode(str), obj, schema);
        return obj;
    }

    public static <T> T fromByte(byte[] data, Class<T> clazz) {
        if (null == data || data.length == 0) {
            return null;
        }
        Schema<T> schema = Inner.getSchema(clazz);
        T obj = schema.newMessage();
        ProtostuffIOUtil.mergeFrom(data, obj, schema);
        return obj;
    }

    public static <T> T fromByte(byte[] data, int offset, int length, Class<T> clazz) {
        if (null == data || data.length == 0) {
            return null;
        }
        Schema<T> schema = Inner.getSchema(clazz);
        T obj = schema.newMessage();
        ProtostuffIOUtil.mergeFrom(data, offset, length, obj, schema);
        return obj;
    }
}
