package io.cana.steam.common.serialize;

import com.fasterxml.jackson.core.json.JsonWriteFeature;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.type.SimpleType;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * @author Dasheng
 * @create 2017-06-07 16:09
 */
public class Jackson {
    public static class Inner {
        public static final String DEFAULT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";

        public static final TypeReference<Map<String, Object>> mapObjReference = new TypeReference<Map<String, Object>>() {
        };

        public static final TypeReference<Map<String, String>> mapStrReference = new TypeReference<Map<String, String>>() {
        };

        /**
         * 默认不排除任何属性
         */
        private static final ObjectMapper MAPPER = new ObjectMapper();

        static {
            MAPPER.writer().with(JsonWriteFeature.ESCAPE_NON_ASCII);
            MAPPER.disable(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS);
            MAPPER.disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);
            MAPPER.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        }

        public static JavaType createMapType(Class<?> keyClass, Class<?> valueClass) {
            return MAPPER.getTypeFactory().constructMapType(Map.class, keyClass, valueClass);
        }

        public static JavaType createArrayType(Class<?> elementClass) {
            return MAPPER.getTypeFactory().constructArrayType(elementClass);
        }

        public static JavaType createSetType(Class<?> elementClass) {
            return MAPPER.getTypeFactory().constructCollectionType(Set.class, elementClass);
        }

        public static JavaType createListType(Class<?> elementClass) {
            return MAPPER.getTypeFactory().constructCollectionType(List.class, elementClass);
        }

        public static JavaType createParametricType(Class<?> parametrized, Class<?>... parameterClasses) {
            return MAPPER.getTypeFactory().constructParametricType(parametrized, parameterClasses);
        }

        public static JavaType createReferenceType(Class<?> rawType, Class<?> referredType) {
            return MAPPER.getTypeFactory().constructReferenceType(rawType, SimpleType.constructUnsafe(referredType));
        }
    }

    public static ObjectMapper getMapper() {
        return Inner.MAPPER;
    }

    public static <T> T beanCopy(Object source, Class<T> target) {
        if (Objects.isNull(source)) {
            return null;
        }
        return getMapper().convertValue(source, target);
    }

    public static <T> T beanCopy(Object source, TypeReference<T> target) {
        if (Objects.isNull(source)) {
            return null;
        }
        return getMapper().convertValue(source, target);
    }

    public static <T> List<T> ListCopy(Object source, Class<T> target) {
        if (Objects.isNull(source)) {
            return null;
        }
        return getMapper().convertValue(source, Inner.createListType(target));
    }

    public static Map<String, Object> bean2Map(Object target) {
        if (Objects.isNull(target)) {
            return null;
        }
        return getMapper().convertValue(target, Inner.mapObjReference);
    }

    public static Map<String, String> bean2StringMap(Object target) {
        if (Objects.isNull(target)) {
            return null;
        }
        return getMapper().convertValue(target, Inner.mapStrReference);
    }

    public static <T> T map2Bean(Map<?, ?> map, Class<T> clazz) {
        if (Objects.isNull(map)) {
            return null;
        }
        return getMapper().convertValue(map, clazz);
    }

    public static Map<String, String> map2StringMap(Map<?, ?> map) {
        if (Objects.isNull(map)) {
            return null;
        }
        return getMapper().convertValue(map, Inner.mapStrReference);
    }

    public static <T> List<T> listConvert(List<?> list, Class<T> clazz) {
        if (Objects.isNull(list)) {
            return null;
        }
        return getMapper().convertValue(list, Inner.createListType(clazz));
    }

    public static String toJson(Object target) {
        if (null == target) {
            return null;
        }
        try {
            return getMapper().writeValueAsString(target);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] toByte(Object target) {
        if (null == target) {
            return null;
        }
        try {
            return getMapper().writeValueAsBytes(target);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static String toPrettyJson(Object target) {
        if (null == target) {
            return null;
        }
        try {
            return getMapper().writerWithDefaultPrettyPrinter().writeValueAsString(target);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] toPrettyByte(Object target) {
        if (null == target) {
            return null;
        }
        try {
            return getMapper().writerWithDefaultPrettyPrinter().writeValueAsBytes(target);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static Map<String, Object> fromJson(String json) {
        if (isBlank(json)) {
            return null;
        }
        try {
            return getMapper().readValue(json, Inner.mapObjReference);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static <T> T fromJson(String json, Class<T> clazz) {
        if (isBlank(json)) {
            return null;
        }
        try {
            return getMapper().readValue(json, clazz);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @SuppressWarnings("unchecked")
    public static <T> T fromJson(String json, JavaType javaType) {
        if (isBlank(json)) {
            return null;
        }
        try {
            return (T) getMapper().readValue(json, javaType);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static <T> T fromJson(String json, TypeReference<T> type) {
        if (isBlank(json)) {
            return null;
        }
        try {
            return getMapper().readValue(json, type);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static <T> T fromJson(String json, Class<T> raw, Class<?> ref) {
        if (isBlank(json)) {
            return null;
        }
        try {
            return getMapper().readValue(json, Inner.createParametricType(raw, ref));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static <T> T fromJson(byte[] bytes, Class<T> clazz) {
        if (null == bytes || bytes.length == 0) {
            return null;
        }
        try {
            return getMapper().readValue(bytes, clazz);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static <T> T fromJson(byte[] bytes, TypeReference<T> type) {
        if (null == bytes || bytes.length == 0) {
            return null;
        }
        try {
            return getMapper().readValue(bytes, type);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static <T> T fromJson(byte[] bytes, Class<T> raw, Class<?> ref) {
        if (null == bytes || bytes.length == 0) {
            return null;
        }
        try {
            return getMapper().readValue(bytes, Inner.createParametricType(raw, ref));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static <E> Set<E> json2Set(String json, Class<E> clazz) {
        if (isBlank(json)) {
            return Collections.emptySet();
        }
        try {
            return getMapper().readValue(json, Inner.createSetType(clazz));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static Set<String> json2StringSet(String json) {
        return json2Set(json, String.class);
    }

    public static <E> List<E> json2List(String json, Class<E> clazz) {
        if (isBlank(json)) {
            return Collections.emptyList();
        }
        try {
            return getMapper().readValue(json, Inner.createListType(clazz));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static List<String> json2StringList(String json) {
        return json2List(json, String.class);
    }

    public static <E> List<E> byte2List(byte[] json, Class<E> clazz) {
        if (null == json) {
            return Collections.emptyList();
        }
        try {
            return getMapper().readValue(json, Inner.createListType(clazz));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static List<String> byte2StringList(byte[] json) {
        return byte2List(json, String.class);
    }

    public static <K, V> Map<K, V> json2Map(String json, Class<K> key, Class<V> value) {
        if (isBlank(json)) {
            return Collections.emptyMap();
        }
        try {
            return getMapper().readValue(json, Inner.createMapType(key, value));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static Map<String, Object> json2Map(String json) {
        return json2Map(json, String.class, Object.class);
    }

    public static <V> Map<String, V> json2Map(String json, Class<V> value) {
        return json2Map(json, String.class, value);
    }

    public static Map<String, String> json2StringMap(String json) {
        return json2Map(json, String.class, String.class);
    }

    public static <K, V> Map<K, V> byte2Map(byte[] json, Class<K> key, Class<V> value) {
        if (null == json) {
            return Collections.emptyMap();
        }
        try {
            return getMapper().readValue(json, Inner.createMapType(key, value));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static Map<String, Object> byte2Map(byte[] json) {
        return byte2Map(json, String.class, Object.class);
    }

    public static <V> Map<String, V> byte2Map(byte[] json, Class<V> value) {
        return byte2Map(json, String.class, value);
    }

    public static Map<String, String> byte2StringMap(byte[] json) {
        return byte2Map(json, String.class, String.class);
    }

    public static boolean isBlank(String str) {
        return str == null || str.trim().isEmpty();
    }
}