package io.bitsensor.plugins.java.http.filter.handler;

import io.bitsensor.plugins.java.core.BitSensor;
import io.bitsensor.proto.shaded.com.google.common.collect.ImmutableMap;
import org.springframework.stereotype.Component;

import javax.servlet.DispatcherType;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

/**
 * Collects information about the HTTP request.
 */
@Component
public class HttpRequestHandler implements RequestHandler {
    private static final Map<String, String> headerNames = createHeaderNames();

    @Override
    public void handle(HttpServletRequest request, HttpServletResponse response) {
        if (request.getDispatcherType().equals(DispatcherType.REQUEST)) {
            BitSensor.addContext("http.version", request.getProtocol());
            BitSensor.addContext("http.query", request.getQueryString());
            BitSensor.addContext("http.type", request.getMethod());
            BitSensor.addContext("http.pathInfo", request.getPathInfo());
            BitSensor.addContext("http.https", Boolean.toString(request.isSecure()));
            BitSensor.addContext("authentication.type", request.getAuthType());
            BitSensor.addContext("authentication.user", request.getRemoteUser());

            Enumeration<String> headers = request.getHeaderNames();
            while (headers.hasMoreElements()) {
                String header = headers.nextElement();
                String value = request.getHeader(header);

                if (headerNames.containsKey(header))
                    header = headerNames.get(header);

                BitSensor.addContext("http." + header, value);
            }

            BitSensor.addEndpoint("address", request.getLocalAddr());
            BitSensor.addEndpoint("name", request.getHeader("Host"));
            BitSensor.addEndpoint("software", request.getServletContext().getServerInfo());
            BitSensor.addEndpoint("port", Integer.toString(request.getServerPort()));
            BitSensor.addEndpoint("uri", request.getRequestURI());
        } else if (request.getDispatcherType().equals(DispatcherType.FORWARD)) {
            BitSensor.addEndpoint("forward.uri", request.getRequestURI());
        }

    }

    @Override
    public ExecutionMoment getExecutionMoment() {
        return ExecutionMoment.PRE_HANDLE;
    }

    private static Map<String, String> createHeaderNames() {
        Map<String, String> headerNames = new HashMap<>();

        headerNames.put("user-agent", "userAgent");
        headerNames.put("accept", "acceptMedia");
        headerNames.put("accept-charset", "acceptCharset");
        headerNames.put("accept-encoding", "acceptEncoding");
        headerNames.put("accept-language", "acceptLanguage");
        headerNames.put("content-type", "contentType");
        headerNames.put("content-length", "contentLength");
        headerNames.put("x-forwarded-for", "x_forwarded_for");
        headerNames.put("x-forwarded-proto", "x_forwarded_proto");

        return ImmutableMap.copyOf(headerNames);
    }
}
