package io.bitsensor.plugins.java.http.filter.handler;

import io.bitsensor.lib.entity.proto.Error;
import io.bitsensor.lib.entity.proto.GeneratedBy;
import io.bitsensor.plugins.java.core.BitSensor;
import io.bitsensor.plugins.java.http.filter.BitSensorRequestWrapper;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Arrays;
import java.util.Map;

/**
 * Collects information about the HTTP request data.
 */
@Component
public class HttpInputHandler implements RequestHandler {

    @Override
    public void handle(HttpServletRequest request, HttpServletResponse response) {
        String method = request.getMethod().toLowerCase();

        handleParameterMap(request, method);
        handleRequestBody(request, method);
    }

    private void handleParameterMap(HttpServletRequest request, String method) {
        for (Map.Entry<String, String[]> valuesByKey : request.getParameterMap().entrySet()) {
            String[] values = valuesByKey.getValue();
            String key = valuesByKey.getKey();

            String valueString = "";
            if (values.length > 1) {
                valueString = Arrays.toString(values);
            } else if (values.length == 1) {
                valueString = values[0];
            }

            BitSensor.addInput("http." + method + "." + key, valueString);
        }
    }

    private void handleRequestBody(HttpServletRequest request, String method) {
        if (!(request instanceof BitSensorRequestWrapper))
            return;

        BitSensorRequestWrapper wrapper = (BitSensorRequestWrapper) request;

        try {
            byte[] bodyBytes = wrapper.getRequestBody();

            // Do not log request body heavier than 100kb
            if (bodyBytes.length > 102400) {
                BitSensor.addError(Error.newBuilder()
                        .setDescription("Request body with size heavier than 100KB is not collected.")
                        .setGeneratedBy(GeneratedBy.PLUGIN)
                        .build());
                return;
            }

            String body = new String(bodyBytes);

            if (!body.trim().isEmpty())
                BitSensor.addInput("http." + method + ".body", body);
        } catch (IOException ignored) {
        }
    }

    @Override
    public ExecutionMoment getExecutionMoment() {
        return ExecutionMoment.POST_HANDLE;
    }
}
