package io.bitsensor.plugins.java.blocking;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.bitsensor.lib.entity.BlockedAttacker;
import io.bitsensor.lib.entity.Identifiable;
import io.bitsensor.plugins.java.blocking.BlockingEvent.Type;
import io.bitsensor.plugins.java.core.BitSensorException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.DirectExchange;
import org.springframework.amqp.rabbit.AsyncRabbitTemplate;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.stereotype.Component;
import org.springframework.util.concurrent.ListenableFutureCallback;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import static io.bitsensor.lib.jackson.JacksonConfig.objectMapper;


/**
 * A simple blocking manager implementation that fetches blacklist from Bitbrain rabbitmq instance and subscribes to
 * blacklist changes.
 */
@Component
public class BlockingManagerImpl implements BlockingManager, ApplicationListener<ContextRefreshedEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(BlockingManager.class);
    private final ApplicationEventPublisher publisher;
    private final DirectExchange rpcExchange;
    private final AsyncRabbitTemplate asyncTemplate;
    private Map<String, BlockedAttacker> blockedAttackerMap = new ConcurrentHashMap<>();
    private ObjectMapper mapper = objectMapper();

    @Autowired
    public BlockingManagerImpl(ApplicationEventPublisher publisher, DirectExchange rpcExchange, RabbitTemplate template, AsyncRabbitTemplate asyncTemplate) {
        this.publisher = publisher;
        this.rpcExchange = rpcExchange;
        this.asyncTemplate = asyncTemplate;
    }

    @Override
    public Map<String, BlockedAttacker> getBlacklist() {
        return blockedAttackerMap;
    }

    @Override
    public BlockedAttacker getBlocking(String id) {
        return blockedAttackerMap.get(id);
    }

    @Override
    public void fetchBlacklist() {
        doFetchBlacklist().addCallback(new ListenableFutureCallback<Object>() {
            @Override
            public void onFailure(Throwable ex) {
                LOGGER.error("Failed fetching blacklist from BitSensor", ex);
                publisher.publishEvent(new BlockingEvent(this, Type.FETCH, false));
            }

            @Override
            public void onSuccess(Object result) {
                if (result == null) {
                    onFailure(new BitSensorException("Blacklist fetch returns null."));
                    return;
                }

                List<Identifiable<BlockedAttacker>> res = mapper.convertValue(result,
                        new TypeReference<List<Identifiable<BlockedAttacker>>>() {
                        });

                blockedAttackerMap.clear();
                res.forEach(i -> blockedAttackerMap.put(i.getIdentifier(), i.getIdentifiable()));

                publisher.publishEvent(new BlockingEvent(this, Type.FETCH, blockedAttackerMap));
            }
        });
    }

    /**
     * Returns a list of {@link BlockedAttacker} being fetched from BitSensor.
     */
    private AsyncRabbitTemplate.RabbitConverterFuture<Object> doFetchBlacklist() {
        String message = "getAll";

        return asyncTemplate.convertSendAndReceive(
                rpcExchange.getName(),
                BlockingConfig.RPC_ROUTING_KEY,
                message);
    }


    @RabbitListener(queues = "#{blacklistAddQueue.name}")
    public void receiveAdd(Identifiable<BlockedAttacker> attacker) throws InterruptedException {
        blockedAttackerMap.put(attacker.getIdentifier(), attacker.getIdentifiable());
        publisher.publishEvent(new BlockingEvent(this, Type.ADD, attacker));
    }


    @RabbitListener(queues = "#{blacklistUpdateQueue.name}")
    public void receiveUpdate(Identifiable<BlockedAttacker> attacker) throws InterruptedException {
        blockedAttackerMap.put(attacker.getIdentifier(), attacker.getIdentifiable());
        publisher.publishEvent(new BlockingEvent(this, Type.UPDATE, attacker));
    }

    @RabbitListener(queues = "#{blacklistDeleteQueue.name}")
    public void receiveDelete(String id) throws InterruptedException {
        if (id == null) {
            return;
        }
        blockedAttackerMap.remove(id);
        publisher.publishEvent(new BlockingEvent(this, Type.DELETE, id));
    }

    @Override
    public void onApplicationEvent(ContextRefreshedEvent event) {
        fetchBlacklist();
    }
}
