package io.bitsensor.plugins.java.blocking;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.*;
import org.springframework.amqp.rabbit.AsyncRabbitTemplate;
import org.springframework.amqp.rabbit.annotation.EnableRabbit;
import org.springframework.amqp.rabbit.config.SimpleRabbitListenerContainerFactory;
import org.springframework.amqp.rabbit.connection.CachingConnectionFactory;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitAdmin;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.rabbit.listener.SimpleMessageListenerContainer;
import org.springframework.amqp.support.converter.Jackson2JsonMessageConverter;
import org.springframework.amqp.support.converter.MessageConverter;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;

import static io.bitsensor.lib.jackson.JacksonConfig.objectMapper;

/**
 * Bean configuration class contains RabbitMQ connection and function configuration as well as exchanges, queue and
 * binding used for getting and listening to blacklist update from bitbrain-es.
 */
@Configuration
@ComponentScan("io.bitsensor.plugins.java.blocking")
@EnableRabbit
public class BlockingConfig {

    public static final String TOPIC_EXCHANGE = "blacklist.topic";
    public static final String ADD_QUEUE = "add";
    public static final String UPDATE_QUEUE = "update";
    public static final String DELETE_QUEUE = "delete";
    public static final String RPC_EXCHANGE = "blacklist.rpc";
    public static final String RPC_REQUEST_QUEUE = "blacklist.rpc.requests";
    public static final String RPC_REPLY_QUEUE = "blacklist.rpc.replies";
    public static final String RPC_ROUTING_KEY = "rpc";
    private static final Logger LOGGER = LoggerFactory.getLogger(BlockingConfig.class);

    /**
     * RabbitMQ configuration
     */
    public static class RabbitMqConfig {

        @Value("${bitsensor.blocking.host:localhost}")
        private String host;

        @Value("${bitsensor.blocking.port:5672}")
        private int port;

        @Value("${bitsensor.blocking.username:guest}")
        private String username;

        @Value("${bitsensor.blocking.password:guest}")
        private String password;

        @Value("${bitsensor.blocking.virtual-host:/}")
        private String virtualHost;

        @Bean
        public ConnectionFactory connectionFactory() {
            CachingConnectionFactory connectionFactory = new CachingConnectionFactory(host, port);
            connectionFactory.setUsername(username);
            connectionFactory.setPassword(password);
            connectionFactory.setVirtualHost(virtualHost);

            LOGGER.info("BitSensor connects: {}@{}:{}/{}", username, host, port, virtualHost);

            return connectionFactory;
        }

        @Bean
        public RabbitAdmin rabbitAdmin(ConnectionFactory connectionFactory) {
            RabbitAdmin rabbitAdmin = new RabbitAdmin(connectionFactory);
            rabbitAdmin.afterPropertiesSet();

            return rabbitAdmin;
        }

        /**
         * Returns a {@code RabbitListenerContainerFactory} configured to have custom message converter {@link
         * #jsonMessageConverter()}. This bean declaration is necessary for converting incoming message.
         */
        @Bean
        public SimpleRabbitListenerContainerFactory rabbitListenerContainerFactory(ConnectionFactory connectionFactory) {
            SimpleRabbitListenerContainerFactory factory = new SimpleRabbitListenerContainerFactory();
            factory.setConnectionFactory(connectionFactory);
            factory.setMessageConverter(jsonMessageConverter());

            return factory;
        }

        /**
         * Returns a {@code MessageConverter} configured to be able to parse Protobuf message and json message.
         */
        @Bean
        public MessageConverter jsonMessageConverter() {
            return new Jackson2JsonMessageConverter(objectMapper());
        }


        @Bean
        public RabbitTemplate rabbitTemplate(ConnectionFactory connectionFactory) {
            RabbitTemplate template = new RabbitTemplate(connectionFactory);
            template.setMessageConverter(jsonMessageConverter());

            return template;
        }

        @Bean
        public AsyncRabbitTemplate asyncRabbitTemplate(ConnectionFactory connectionFactory, RabbitTemplate rabbitTemplate) {
            SimpleMessageListenerContainer listenerContainer = new SimpleMessageListenerContainer(connectionFactory);
            listenerContainer.setQueueNames(RPC_REPLY_QUEUE);
            AsyncRabbitTemplate template = new AsyncRabbitTemplate(rabbitTemplate, listenerContainer);

            return template;
        }
    }

    /**
     * Client configuration for exchanges, queues and bindings. Note that exchanges, queues and bindings here will not
     * be automatically declared and appear in RabbitMq without RabbitAdmin bean declared.
     */
    public static class ClientConfig {

        @Bean
        public DirectExchange rpcExchange() {
            return new DirectExchange(RPC_EXCHANGE);
        }

        @Bean
        public Queue rpcReplyQueue() {
            return new Queue(RPC_REPLY_QUEUE);
        }

        @Bean
        public Queue rpcRequestQueue() {
            return new Queue(RPC_REQUEST_QUEUE);
        }

        @Bean
        public Binding binding(DirectExchange directExchange, Queue rpcRequestQueue) {
            return BindingBuilder.bind(rpcRequestQueue).to(directExchange).with(RPC_ROUTING_KEY);
        }

        @Bean
        public TopicExchange topicExchange() {
            return new TopicExchange(TOPIC_EXCHANGE);
        }

        @Bean
        public Queue blacklistAddQueue() {
            return new AnonymousQueue();
        }

        @Bean
        public Queue blacklistDeleteQueue() {
            return new AnonymousQueue();
        }

        @Bean
        public Queue blacklistUpdateQueue() {
            return new AnonymousQueue();
        }

        @Bean
        public Binding binding1(TopicExchange topicExchange, Queue blacklistAddQueue) {
            return BindingBuilder.bind(blacklistAddQueue).to(topicExchange).with(ADD_QUEUE);
        }

        @Bean
        public Binding binding2(TopicExchange topicExchange, Queue blacklistUpdateQueue) {
            return BindingBuilder.bind(blacklistUpdateQueue).to(topicExchange).with(UPDATE_QUEUE);
        }

        @Bean
        public Binding binding3(TopicExchange topicExchange, Queue blacklistDeleteQueue) {
            return BindingBuilder.bind(blacklistDeleteQueue).to(topicExchange).with(DELETE_QUEUE);
        }
    }
}
