package io.bitsensor.lib.util;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.util.ISO8601DateFormat;
import com.fasterxml.jackson.databind.util.ISO8601Utils;
import io.bitsensor.lib.jackson.protobuf.ProtobufModule;
import io.bitsensor.proto.shaded.com.google.protobuf.util.JsonFormat;
import org.joda.time.DateTime;
import org.joda.time.Interval;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.integration.support.json.Jackson2JsonObjectMapper;

import java.io.IOException;
import java.text.FieldPosition;
import java.util.AbstractMap.SimpleEntry;
import java.util.Date;

@Configuration
public class JacksonConfig {

    @Bean
    public static Module configureModule() {
        SimpleModule module = new SimpleModule();
        module.addSerializer(SimpleEntry.class, new AbstractMapSimpleEntrySerializer());
        module.addSerializer(Interval.class, new IntervalSerializer());
        module.addSerializer(DateTime.class, new JodaDateTimeAsMillisecondsSerializer());
        module.setMixInAnnotation(SimpleEntry.class, Mixin.class);
        return module;
    }

    /**
     * Installs Protobuf jackson module for parsing protobuf message.
     */
    @Bean
    public static Module protobufModule() {
        return new ProtobufModule(
                JsonFormat.printer()
                        .omittingInsignificantWhitespace()
                        .includingDefaultValueFields()
                        .preservingProtoFieldNames(),
                JsonFormat.parser()
                        .ignoringUnknownFields()
        );
    }

    @Bean
    public static ObjectMapper objectMapper() {
        return new ObjectMapper()
                .registerModule(configureModule())
                .registerModule(protobufModule())
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
                .configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false)
                .setDateFormat(new ISO8601DateFormatMilliseconds());
    }

    @Bean
    public static Jackson2JsonObjectMapper jackson2JsonObjectMapper() {
        return new Jackson2JsonObjectMapper(objectMapper());
    }

    abstract static class Mixin<K, V> {
        public Mixin(@JsonProperty("key") K key, @JsonProperty("value") V value) {
        }
    }

    public static class AbstractMapSimpleEntrySerializer extends JsonSerializer<SimpleEntry> {
        @Override
        public void serialize(SimpleEntry simpleEntry, JsonGenerator jsonGenerator, SerializerProvider serializerProvider) throws IOException {
            jsonGenerator.writeStartObject();
            jsonGenerator.writeObjectField("key", simpleEntry.getKey());
            jsonGenerator.writeObjectField("value", simpleEntry.getValue());
            jsonGenerator.writeEndObject();
        }
    }

    public static class IntervalSerializer extends JsonSerializer<Interval> {
        @Override
        public void serialize(Interval interval, JsonGenerator jsonGenerator, SerializerProvider serializerProvider) throws IOException {
            jsonGenerator.writeStartObject();
            jsonGenerator.writeObjectField("start", interval.getStart());
            jsonGenerator.writeObjectField("end", interval.getEnd());
            jsonGenerator.writeEndObject();
        }
    }

    public static class JodaDateTimeAsMillisecondsSerializer extends JsonSerializer<DateTime> {
        @Override
        public void serialize(DateTime dateTime, JsonGenerator jsonGenerator, SerializerProvider serializerProvider) throws IOException {
            jsonGenerator.writeNumber(dateTime.getMillis());
        }
    }

    public static class ISO8601DateFormatMilliseconds extends ISO8601DateFormat {
        @Override
        public StringBuffer format(Date date, StringBuffer toAppendTo, FieldPosition fieldPosition) {
            String value = ISO8601Utils.format(date, true);
            toAppendTo.append(value);
            return toAppendTo;
        }
    }
}

