package io.bitsensor.lib.util;

import com.google.gson.*;
import io.bitsensor.lib.util.exception.DataModelException;

import java.util.Map;

import static java.lang.String.join;
import static java.util.Arrays.asList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * InputConverter class for converting a json object or json string into a flattened/unflattened json object.
 */
public class InputConverter {
    private static final Logger LOGGER = LoggerFactory.getLogger(InputConverter.class);

    /**
     * Returns unflattened json object.
     *
     * @param json input json string
     * @return json object
     */
    public static JsonObject unflatten(String json) throws DataModelException {
        return unflatten(json, new String[]{});
    }

    /**
     * Returns unflattened json object which has {@code excludeFields} flattened
     *
     * @param json          input json string
     * @param excludeFields fields to exclude
     * @return json object
     */
    public static JsonObject unflatten(String json, String[] excludeFields) throws DataModelException {
        JsonElement partlyFlattenedObject = new JsonParser().parse(json);

        try {
            return unflatten(partlyFlattenedObject, excludeFields);
        } catch (DataModelException ex) {
            LOGGER.error(json, ex);
            throw ex;
        }
    }

    /**
     * Returns unflattened json object.
     *
     * @param element input json element
     * @return json object
     */
    public static JsonObject unflatten(JsonElement element) throws DataModelException {
        return unflatten(element, new String[]{});
    }

    /**
     * Returns unflattened json object which has {@code excludeFields} flattened
     *
     * @param element       input json element
     * @param excludeFields fields to exclude
     * @return json object
     */
    public static JsonObject unflatten(JsonElement element, String[] excludeFields) throws DataModelException {
        JsonObject flattened = flatten(element);

        JsonObject jsonObject = new JsonObject();
        for (Map.Entry<String, JsonElement> nameElementEntry : flattened.getAsJsonObject().entrySet()) {
            String name = nameElementEntry.getKey();
            JsonElement value = nameElementEntry.getValue();
            unflattenRaw(name.split("\\."), value, jsonObject, excludeFields, true);
        }

        return jsonObject;
    }

    public static void unflatten(String[] parts, String value, JsonObject parent) throws DataModelException {
        unflattenRaw(parts, (value != null) ? new JsonPrimitive(value) : new JsonNull(), parent, new String[]{}, true);
    }

    public static void unflattenRaw(String[] parts, JsonElement value, JsonObject parent, String[] excludeFields, boolean recursive) throws DataModelException {
        if (parts.length > 0) {
            String key = parts[0];
            parts = removeFirst(parts);

            if (parts.length == 0) {
                if (value.isJsonArray()) {
                    JsonArray newArray = new JsonArray();
                    for (JsonElement element : value.getAsJsonArray()) {
                        newArray.add(
                                element.isJsonObject()
                                        ? unflatten(element, excludeFields)
                                        : element);
                    }
                    value = newArray;
                }
                parent.add(key, value);
            } else {
                if (parent.has(key)) {
                    JsonElement jsonElement = parent.get(key);

                    //If there exists a value, and a nested key needs to be added
                    if (jsonElement.isJsonPrimitive()) {
                        String valueToNest = jsonElement.getAsString();

                        JsonObject nestedObject = new JsonObject();

                        //prefix that value with _
                        nestedObject.addProperty("_", valueToNest);

                        //and re add the value as JsonObject with its value nested
                        parent.remove(key);
                        parent.add(key, nestedObject);

                        jsonElement = parent.get(key);
                    }

                    if (jsonElement.isJsonObject()) {
                        unflattenRaw(parts, value, jsonElement.getAsJsonObject(), excludeFields, recursive && !asList(excludeFields).contains(key));
                    } else {
                        throw new DataModelException("An object in the data model is also assigned a Array value. " +
                                key + "." + join(".", parts) + ": " + jsonElement.toString());
                    }
                } else if (!recursive) {
                    parent.add(key + "." + join(".", parts), value);
                } else {
                    JsonObject jsonObject = new JsonObject();
                    unflattenRaw(parts, value, jsonObject, excludeFields, recursive && !asList(excludeFields).contains(key));
                    parent.add(key, jsonObject);
                }
            }
        }
    }

    /**
     * Returns flattened json object.
     *
     * @param json input json string
     * @return json object
     */
    public static JsonObject flatten(String json) {
        JsonElement parsedJson = new JsonParser().parse(json);

        return flatten(parsedJson);
    }

    /**
     * Returns flattened json object.
     *
     * @param element input json element
     * @return json object
     */
    public static JsonObject flatten(JsonElement element) {
        JsonObject jsonObject = new JsonObject();
        flatten(element, "", jsonObject);

        return jsonObject;
    }

    public static void flatten(JsonElement object, String prefix, JsonObject parent) {
        if (object.isJsonObject()) {
            if (object.getAsJsonObject().entrySet().size() == 0) {
                if (!prefix.equals(""))
                    parent.add(prefix, object);
            }
            for (Map.Entry<String, JsonElement> entry : object.getAsJsonObject().entrySet()) {
                flatten(entry.getValue(), (prefix.isEmpty() ? "" : prefix + ".") + entry.getKey(), parent);
            }
        } else {
            if (object.isJsonArray()) {
                JsonArray newArray = new JsonArray();

                for (JsonElement element : object.getAsJsonArray()) {
                    newArray.add(
                            element.isJsonObject()
                                    ? flatten(element)
                                    : element);
                }
                object = newArray;
            }
            parent.add(prefix, object);
        }
    }

    private static String[] removeFirst(String[] parts) {
        String[] strippedParts = new String[parts.length - 1];
        System.arraycopy(parts, 1, strippedParts, 0, parts.length - 1);
        return strippedParts;
    }

}
