package io.bitexpress.openapi.client;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Setter;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.exception.ContextedRuntimeException;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.file.Path;
import java.util.Arrays;

public class GatewayClientBuilder extends BaseBuilder {
    @Setter
    private String url;

    @Setter
    private ClientHttpRequestFactory clientHttpRequestFactory;

    @Setter
    private Resource pkcs1PrivateKeyPairResource;

    @Setter
    private Resource pkcs8PrivateKeyResource;

    public GatewayClient build() {
        GatewayClientImpl client = new GatewayClientImpl();
        client.setUrl(Validate.notBlank(url, "url is blank"));

        ObjectMapper objectMapper = createObjectMapper();
        client.setObjectMapper(objectMapper);
        client.setRequestFactory(createRequestFactory(objectMapper));
        client.setRestTemplate(createRestTemplate(objectMapper));
        return client;
    }


    private RequestFactory createRequestFactory(ObjectMapper objectMapper) {
        RequestFactory requestFactory = new RequestFactory();
        requestFactory.setObjectMapper(objectMapper);
        requestFactory.setUid(uid);
        requestFactory.setIssuerCode(issuerCode);
        requestFactory.setIssuer(issuer);
        if (pkcs1PrivateKeyPairResource != null) {
            try (Reader reader = new InputStreamReader(pkcs1PrivateKeyPairResource.getInputStream())) {
                requestFactory.setPrivateKey(keyReader.readPkcs1KeyPair(reader).getLeft());
            } catch (IOException e) {
                throw new ContextedRuntimeException(e);
            }
        } else if (pkcs8PrivateKeyResource != null) {
            try (Reader reader = new InputStreamReader(pkcs8PrivateKeyResource.getInputStream())) {
                requestFactory.setPrivateKey(keyReader.readPkcs8PrivateKey(reader));
            } catch (IOException e) {
                throw new ContextedRuntimeException(e);
            }
        } else {
            throw new IllegalArgumentException("pkcs1PrivateKeyPairResource and pkcs8PrivateKeyResource are both null.");
        }
        return requestFactory;
    }

    private RestTemplate createRestTemplate(ObjectMapper objectMapper) {
        MappingJackson2HttpMessageConverter mappingJackson2HttpMessageConverter = new WriteOnlyMappingJackson2HttpMessageConverter(
                objectMapper);
        StringHttpMessageConverter stringHttpMessageConverter = new StringHttpMessageConverter();
        if (clientHttpRequestFactory == null) {
            clientHttpRequestFactory = new HttpComponentsClientHttpRequestFactory();
        }
        RestTemplate restTemplate = new RestTemplate(clientHttpRequestFactory);
        restTemplate
                .setMessageConverters(Arrays.asList(mappingJackson2HttpMessageConverter, stringHttpMessageConverter));
        return restTemplate;
    }


    @Deprecated
    public void setClientPrivateKeyFilePath(Path path) {
        setClientPrivateKeyResource(new FileSystemResource(path));
    }

    @Deprecated
    public void setClientPrivateKeyResource(Resource resource) {
        setPkcs1PrivateKeyPairResource(resource);
    }

    public void setPkcs1PrivateKeyPairFilePath(Path path) {
        setPkcs1PrivateKeyPairResource(new FileSystemResource(path));
    }

    public void setPkcs8PrivateKeyFilePath(Path path) {
        setPkcs8PrivateKeyResource(new FileSystemResource(path));
    }
}
