package io.bitexpress.openapi.client.notification;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.bitexpress.openapi.client.FromHolder;
import io.bitexpress.openapi.client.sign.OaRsaSignatureTool;
import io.bitexpress.openapi.model.ack.NotificationAck;
import io.bitexpress.openapi.model.content.OpenApiRequestContent;
import io.bitexpress.openapi.model.content.OpenApiRequestHeader;
import io.bitexpress.openapi.model.content.ServiceIndex;
import io.bitexpress.openapi.model.envelope.OpenApiEnvelope;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.exception.ContextedRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.security.PublicKey;
import java.util.function.Function;

public class GatewayNotificationParserImpl extends FromHolder implements GatewayNotificationParser {

    private static final Logger logger = LoggerFactory.getLogger(GatewayNotificationParserImpl.class);


    private ObjectMapper objectMapper;

    private PublicKey serverSidePublicKey;

    private Function<ServiceIndex, JavaType> notificationJavaTypeFunction;

    @Override
    public <NOTIFY> OpenApiRequestContent<NOTIFY> parseNotification(String requestEnvelope) {
        OpenApiEnvelope openApiEnvelope = null;
        try {
            openApiEnvelope = objectMapper.readValue(requestEnvelope, OpenApiEnvelope.class);

            // 鉴权,如果鉴权失败则抛异常,终止执行
            checkPrivilige(openApiEnvelope.getUid(), openApiEnvelope.getIssuerCode());

            // 验签,如果验签失败则抛异常,终止执行
            verifySignature(openApiEnvelope);

            // 反序列请求内容
            OpenApiRequestContent<JsonNode> openApiRequestContent = unmarshallRequestContent(
                    openApiEnvelope.getContent());
            JavaType javaType = notificationJavaTypeFunction.apply(openApiRequestContent.getHeader().getServiceIndex());
            Validate.notNull(javaType, "javaType not found");
            return unmarshallBody(openApiRequestContent, javaType);
        } catch (IOException e) {
            throw new ContextedRuntimeException(e);
        }
    }

    private void checkPrivilige(Long uid, String issuerCode) {
        Validate.isTrue(this.uid.equals(uid) || this.issuerCode.equals(issuerCode), "from mismatch,your uid:%s, incoming uid:%s,your issuerCode:%s, incoming issuerCode:%s", this.uid, uid, this.issuerCode, issuerCode);
    }

    private void verifySignature(OpenApiEnvelope openApiEnvelope) {
        if (serverSidePublicKey == null) {
            logger.warn("skip verify signature");
            return;
        }
        boolean verify = OaRsaSignatureTool.verifyUtf8WithHex(openApiEnvelope.getContent(),
                openApiEnvelope.getSignature(), serverSidePublicKey);
        Validate.isTrue(verify, "verify signature failure.");
    }

    private OpenApiRequestContent<JsonNode> unmarshallRequestContent(String content) throws IOException {
        JsonNode rootNode = objectMapper.readTree(content);
        JsonNode headerNode = rootNode.get("header");
        OpenApiRequestHeader openApiRequestHeader = objectMapper.treeToValue(headerNode, OpenApiRequestHeader.class);
        OpenApiRequestContent<JsonNode> openApiRequestContent = new OpenApiRequestContent<>();
        openApiRequestContent.setHeader(openApiRequestHeader);
        openApiRequestContent.setBody(rootNode.get("body"));
        return openApiRequestContent;

    }

    private <T> OpenApiRequestContent<T> unmarshallBody(OpenApiRequestContent<JsonNode> requestContent,
                                                        JavaType bodyType) {
        try {
            JsonParser jsonParser = objectMapper.treeAsTokens(requestContent.getBody());
            T t = objectMapper.readValue(jsonParser, bodyType);
            OpenApiRequestContent<T> requestContent2 = new OpenApiRequestContent<>();
            requestContent2.setHeader(requestContent.getHeader());
            requestContent2.setBody(t);
            return requestContent2;
        } catch (IOException e) {
            throw new ContextedRuntimeException(e);
        }

    }

    @Override
    public String getAckString(boolean success) {
        try {
            return objectMapper.writeValueAsString(getAck(success));
        } catch (JsonProcessingException e) {
            throw new ContextedRuntimeException(e);
        }
    }

    @Override
    public NotificationAck getAck(boolean success) {
        NotificationAck ack = new NotificationAck();
        ack.setSuccess(success);
        return ack;
    }

    public ObjectMapper getObjectMapper() {
        return objectMapper;
    }

    public void setObjectMapper(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }


    public void setServerSidePublicKey(PublicKey serverSidePublicKey) {
        this.serverSidePublicKey = serverSidePublicKey;
    }

    public void setNotificationJavaTypeFunction(Function<ServiceIndex, JavaType> notificationJavaTypeFunction) {
        this.notificationJavaTypeFunction = notificationJavaTypeFunction;
    }

}
