package io.bitexpress.openapi.client.notification;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.bitexpress.openapi.client.BaseBuilder;
import io.bitexpress.openapi.model.content.ServiceIndex;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.exception.ContextedRuntimeException;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.security.PublicKey;
import java.util.function.Function;


public abstract class AbstractGatewayNotificationParserBuilder extends BaseBuilder {

    private Resource serverSidePublicKeyResource;

    public GatewayNotificationParser build() {
        ObjectMapper objectMapper = createObjectMapper();
        GatewayNotificationParserImpl gatewayNotificationParser = new GatewayNotificationParserImpl();
        gatewayNotificationParser.setObjectMapper(objectMapper);
        gatewayNotificationParser.setUid(uid);
        gatewayNotificationParser.setIssuerCode(issuerCode);

        gatewayNotificationParser.setServerSidePublicKey(readPublicKey());
        Function<ServiceIndex, JavaType> notificationJavaTypeFunction = getTypeFunction(objectMapper);
        gatewayNotificationParser.setNotificationJavaTypeFunction(notificationJavaTypeFunction);
        return gatewayNotificationParser;
    }

    private PublicKey readPublicKey() {
        try (InputStream is = serverSidePublicKeyResource.getInputStream()) {
            String publicKeyString = IOUtils.toString(is, StandardCharsets.US_ASCII);
            return keyReader.readPublicKey(publicKeyString);
        } catch (IOException e) {
            throw new ContextedRuntimeException(e);
        }
    }


    public void setServerSidePublicKeyFilePath(Path serverSidePublicKeyFilePath) {
        setServerSidePublicKeyResource(new FileSystemResource(serverSidePublicKeyFilePath));
    }

    public void setServerSidePublicKeyResource(Resource serverSidePublicKeyResource) {
        this.serverSidePublicKeyResource = serverSidePublicKeyResource;
    }

    protected abstract Function<ServiceIndex, JavaType> getTypeFunction(ObjectMapper objectMapper);

}
